(ns app.routes
  (:require [app.parser :as p]
            [om.next :as om :refer-macros [defui]]
            [sablono.core :refer-macros [html]]
            [tripod.core :as tripod]
            [app.dashboards :as dashboards]
            [app.artists :as artists]
            [app.artworks :as artworks]))

(enable-console-print!)

;; Parsing

(defmethod p/read :route/info
  [{:keys [state query]} k _]
  (let [st @state
        route (get st :app/route)
        route (cond-> route
                      (= (second route) '_) pop)]
    {:value (get-in st route)}))

(defmethod p/read :app/route
  [{:keys [state query]} k _]
  (let [st @state]
    {:value (get st k)}))

(defmethod p/mutate 'route/change!
  [{:keys [state]} _ {:keys [route]}]
  {:value  {:keys [:app/route]}
   :action #(swap! state assoc :app/route route)})

;; Plumbing

(def route->component
  {:app/home     dashboards/Dashboard
   :app/artists  artists/ArtistList
   :app/artworks artworks/ArtworkList})

(def route->factory
  (zipmap (keys route->component)
          (map om/factory (vals route->component))))

(def route->query
  (zipmap (keys route->component)
          (map om/get-query (vals route->component))))

(defn change-route! [route]
  (om/transact! p/reconciler `[(~'route/change! {:route ~route})]))

;; Tripod

(defn page [route-name]
  {:name  route-name
   :enter (fn [context] (assoc context :response [route-name '_]))})

(def routes
  (tripod/expand-routes
    [["/" (page :app/home)
      ["/artists" (page :app/artists)]
      ["/artworks" (page :app/artworks)]]]))

(def service
  (-> {::tripod/routes routes}
      tripod/default-interceptors
      tripod/service))

;; UI

(defui Menu
  Object
  (render [this]
    (let [route (om/props this)]
      (html
        [:div {:style {:display         "inline-block"
                       :width           "25%"
                       :minHeight       "80%"
                       :verticalAlign   "top"
                       :backgroundColor "gainsboro"}}
         [:div
          [:ul (for [[title link] [["Home" :app/home]
                                   ["Artists" :app/artists]
                                   ["Artworks" :app/artworks]]]
                 [:li {:key   title
                       :style {:margin "10px"}}
                  [:a {:href  (str "#" (tripod/path-for link))
                       :style (when (= (first route) link)
                                {:color "black"})} title]])]
          [:p {:style {:textAlign      "center"
                       :textDecoration "underline"
                       :marginBottom   "5px"
                       :marginTop      "30px"
                       :fontWeight     "bold"}} "Current route:"]
          [:p {:style {:textAlign "center"
                       :margin    0
                       :color     "red"}} (pr-str route)]]]))))


(def menu (om/factory Menu))

(defui Root
  static om/IQuery
  (query [this]
    [:app/route {:route/info route->query}])
  Object
  (render [this]
    (let [{:keys [app/route route/info]} (om/props this)]
      (html
        [:div {:style {:margin          "0 auto"
                       :height          250
                       :width           500
                       :backgroundColor "oldlace"}}
         [:div {:style {:minWidth          "100%"
                        :minHeight         "48px"
                        :lineHeight        "48px"
                        :verticalAlign     "middle"
                        :borderBottomWidth "2px"
                        :borderBottomStyle "solid"}}
          [:h2 {:style {:margin     0
                        :textAlign  "center"
                        :lineHeight "48px"}} "Om Next Routing"]]
         (menu route)
         [:div {:style {:display         "inline-block"
                        :width           "70%"
                        :minHeight       "80%"
                        :verticalAlign   "top"
                        :padding         "12.5px 12.5px 12.5px 10.5px"
                        :borderLeftWidth "2px"
                        :borderLeftStyle "solid"}}
          ((route->factory (first route)) info)]]))))


