(ns app.routes
  (:require [app.parser :as p]
            [om.dom :as dom]
            [om.next :as om :refer-macros [defui]]
            [sablono.core :refer-macros [html]]
            [tripod.core :as tripod]
            [app.artists :as artists]
            [app.artworks :as artworks]))

(defmethod p/read :route/info
  [{:keys [state query]} k _]
  (let [st @state
        route (get st :app/route)
        route (cond-> route
                      (= (second route) '_) pop)]
    {:value (get-in st route)}))

(defmethod p/read :app/route
  [{:keys [state query]} k _]
  (let [st @state]
    {:value (get st k)}))

(defmethod p/mutate 'route/change!
  [{:keys [state]} _ {:keys [route]}]
  {:value  {:keys [:app/route]}
   :action #(swap! state assoc :app/route route)})

(def route->component
  {:app/home  artists/ArtistList
   :app/about artworks/ArtworkList})

(def route->factory
  (zipmap (keys route->component)
          (map om/factory (vals route->component))))

(def route->query
  (zipmap (keys route->component)
          (map om/get-query (vals route->component))))

(defn- change-route [c route e]
  (.preventDefault e)
  (om/transact! c `[(~'route/change! {:route ~route})]))

(defn change-route! [route]
  (om/transact! p/reconciler `[(~'route/change! {:route ~route})]))

(defui Root
  static om/IQuery
  (query [this]
    [:app/route {:route/info route->query}])
  Object
  (render [this]
    (let [{:keys [app/route route/info]} (om/props this)]
      (dom/div #js {:style #js {:margin          "0 auto"
                                :height          250
                                :width           500
                                :backgroundColor "oldlace"}}
               (dom/div #js {:style #js {:minWidth          "100%"
                                         :minHeight         "48px"
                                         :lineHeight        "48px"
                                         :verticalAlign     "middle"
                                         :borderBottomWidth "2px"
                                         :borderBottomStyle "solid"}}
                        (dom/h2 #js {:style #js {:margin     0
                                                 :textAlign  "center"
                                                 :lineHeight "48px"}}
                                "Om Next Routing"))
               (dom/div #js {:style #js {:display         "inline-block"
                                         :width           "25%"
                                         :minHeight       "80%"
                                         :verticalAlign   "top"
                                         :backgroundColor "gainsboro"}}
                        (dom/ul nil
                                (dom/li #js {:style #js {:marginTop "20px"}}
                                        (dom/a #js {:href    "#/"
                                                    :style   (when (= (first route) :app/home)
                                                               #js {:color  "black"
                                                                    :cursor "text"})}
                                               "Home"))
                                (dom/li #js {:style #js {:marginTop "5px"}}
                                        (dom/a #js {:href    "#/about"
                                                    :style   (when (= (first route) :app/about)
                                                               #js {:color  "black"
                                                                    :cursor "text"})}
                                               "About")))
                        (dom/p #js {:style #js {:textAlign      "center"
                                                :textDecoration "underline"
                                                :marginBottom   "5px"
                                                :marginTop      "30px"
                                                :fontWeight     "bold"}}
                               "Current route:")
                        (dom/p #js {:style #js {:textAlign "center"
                                                :margin    0
                                                :color     "red"}}
                               (str (pr-str route))))
               (dom/div #js {:style #js {:display         "inline-block"
                                         :width           "70%"
                                         :minHeight       "80%"
                                         :verticalAlign   "top"
                                         :padding         "12.5px 12.5px 12.5px 10.5px"
                                         :borderLeftWidth "2px"
                                         :borderLeftStyle "solid"}}
                        ((route->factory (first route)) info))))))

(defn page [id]
  {:name id
   :enter (fn [context] (assoc context :response [id '_]))})

(def routes
  (tripod/expand-routes
    [["/" (page :app/home)
      ["/about" (page :app/about)]]]))

(def service
  (-> {::tripod/routes routes}
      tripod/default-interceptors
      tripod/service))