(ns clojurewerkz.neocons.rest
  (:import  [java.net URI])
  (:require [clj-http.client   :as http]
            [clojure.data.json :as json])
  (:use     [clojurewerkz.neocons.rest.statuses]
            [clojurewerkz.neocons.rest.helpers :only [maybe-append]]))

;;
;; Implementation
;;

(defn GET
  [^String uri  ^String user ^String pass & {:as options }]
  (io!
   (http/get uri (merge options { :accept :json} {:basic-auth [user pass]}))))

(defn POST
  [^String uri ^String user ^String pass &{ :keys [body] :as options }]
  (io!
   (http/post uri (merge options { :accept :json :content-type :json :body body } {:basic-auth [user pass]}))))

(defn PUT
  [^String uri ^String user ^String pass &{ :keys [body] :as options }]
  (io!
   (http/put uri (merge options { :accept :json :content-type :json :body body } {:basic-auth [user pass]}))))

(defn DELETE
  [^String uri ^String user ^String pass &{ :keys [body] :as options }]
  (io!
   (http/delete uri (merge options { :accept :json } {:basic-auth [user pass]}))))




(defrecord Neo4JEndpoint
    [version node-uri relationships-uri node-index-uri relationship-index-uri relationship-types-uri batch-uri extensions-info-uri extensions reference-node-uri uri user pass])

(def ^{ :dynamic true } *endpoint*)

;;
;; API
;;

(defprotocol Connection
  (connect  [uri user pass] "Connects to given Neo4J REST API endpoint and performs service discovery")
  (connect! [uri user pass] "Connects to given Neo4J REST API endpoint, performs service discovery and mutates *endpoint* state to store it"))

(extend-protocol Connection
  URI
  (connect [uri user pass]
    (connect (.toString uri) (.toString user) (.toString pass)))
  (connect! [uri user pass]
    (connect! (.toString uri) (.toString user) (.toString pass)))

  String
  (connect [uri ^String user ^String pass]
    (let [{ :keys [status body] } (GET uri user pass)]
      (if (success? status)
        (let [payload (json/read-json body true)]
          (Neo4JEndpoint. (:neo4j_version      payload)
                          (:node               payload)
                          (str uri (if (.endsWith uri "/")
                                     "relationship"
                                     "/relationship"))
                          (:node_index         payload)
                          (:relationship_index payload)
                          (:relationship_types payload)
                          (:batch              payload)
                          (:extensions_info    payload)
                          (:extensions         payload)
                          (:reference_node     payload)
                          (maybe-append uri "/")
                          (str  user)
                          (str     pass) 
                          ;(user)
                         ; (pass)
                          )))))
  (connect! [uri user pass]
    (defonce ^{ :dynamic true } *endpoint* (connect uri user pass))))
