(ns sock.task.hybrid
  "Sock Boot tasks for hybrid Clojure and Java projects."
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.init]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.clj :as task.clj]
    [sock.task.deps :refer [deps test-deps]]
    [sock.task.java :as task.java])
  (:refer-clojure :exclude [test]))

(core/deftask build
  "Resolve dependencies, compile Java sources, perform AOT compilation on
   Clojure namespaces, create project pom.xml file and build a jar file."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   e env        FOO=BAR {kw str} "Environ environment map."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   _ pom        OPTIONS edn "Options for built-in/pom task."]
  (comp (apply-kw deps tools-deps)
        (task.clj/environ :env env)
        (apply-kw built-in/javac javac)
        (apply-kw built-in/aot aot)
        (apply-kw built-in/pom pom)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Resolve dependencies including :test alias, compile Java sources, perform
   AOT compilation on Clojure namespaces, run Midje tests and run jUnit tests."
  [_ tools-deps OPTIONS edn "seancorfield/boot-tools-deps task."
   e env        FOO=BAR {kw str} "Environ environment map."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   c class-names CLASSNAME #{str} "The set of Java class names to run tests from."
   p packages    PACKAGE   #{str} "The set of package names to run tests from."]
  (comp (test-deps :tools-deps tools-deps)
        (task.clj/environ :env env)
        (apply-kw built-in/javac javac)
        (apply-kw built-in/aot aot)
        (task.clj/midje)
        (task.clj/clojure-test)
        (task.java/junit :class-names class-names
                         :packages packages)))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps tools-deps
               :javac      javac
               :aot        aot
               :pom        pom)
        (apply-kw built-in/jar jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :aot        aot
                 :pom        pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :aot        aot
                 :pom        pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask assemble
  "Build, add jar entries from dependencies to fileset and writes output files
   to the target directory. Mainly used for server projects which subsequently
   copy the target directory into a Docker image."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ aot        OPTIONS edn "Options for built-in/aot task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ uber       OPTIONS edn "Options for built-in/uber task."
   _ target     OPTIONS edn "Options for built-in/target task."]
  (comp (build :tools-deps tools-deps
               :javac      javac
               :aot        aot
               :pom        pom)
        (apply-kw built-in/uber uber)
        (apply-kw built-in/target target)))
