(ns sock.task.plumbing.deps
  "Sock plumbing for dependencies"
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.lib.core :refer [apply-kw]]
    [boot-tools-deps.core :refer [load-deps]]))

(core/deftask deps
  "Use tools.deps to read and resolve the specified deps.edn files.

  The dependencies read in are added to your Boot :dependencies vector.

  With the exception of -A, and -v, the arguments are intended to match
  the clj script usage (as passed to clojure.tools.deps.alpha.makecp/-main).
  Note, in particular, that -c / --config-paths is assumed to be the COMPLETE
  list of EDN files to read (and therefore overrides the default set of
  system deps and local deps)."
  [c config-paths    PATH [str] "the list of deps.edn files to read"
   A aliases           KW [kw]  "the list of aliases (for both -C and -R)"
   C classpath-aliases KW [kw]  "the list of classpath aliases to use"
   R resolve-aliases   KW [kw]  "the list of resolve aliases to use"
   v verbose              bool  "Be verbose (and ask tools.deps to be verbose too)"]
  (let [opts {:config-paths        config-paths
              :classpath-aliases   (into (vec aliases) classpath-aliases)
              :resolve-aliases     (into (vec aliases) resolve-aliases)
              :overwrite-boot-deps true
              :verbose             verbose}]
    (load-deps opts)
    identity))

(core/deftask test-alias-deps
  "Configure the boot environment with typical defaults for testing."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."]
  (let [tools-deps' (merge {:aliases [:test]} tools-deps)]
       (apply-kw deps tools-deps')
       identity))

(core/deftask conflicts
  "Print graph of dependency conflicts."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :pedantic true)))

(core/deftask fake-classpath
  "Print the project's fake classpath."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :fake-classpath true)))

(core/deftask classpath
  "Print the project's classpath."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :classpath true)))

(core/deftask graph
  "Print the project's dependency graph."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."
   v verify-deps        bool "Include signature status of each dependency in graph."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :deps true :verify-deps verify-deps)))

(core/deftask env
  "Print the project's boot env map."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :env true)))

(core/deftask updates
  "Print newer releases of outdated dependencies."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   t test               bool "Use the test-deps task for dependency resolution."
   S snapshots          bool "Include snapshot versions in update searches."]
  (comp
    (apply-kw (if test test-alias-deps deps) tools-deps)
    (built-in/show :updates (not snapshots) :update-snapshots snapshots)))

