(ns sock.task.porcelain.clj
  "Sock porcelain boot tasks for Clojure"
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.init]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.plumbing.deps :as plumbing.deps]
    [sock.task.plumbing.clj :as plumbing.clj])
  (:refer-clojure :exclude [test]))

(core/deftask build
  "Resolve dependencies, create project pom.xml file and build a jar file."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."]
  (comp (apply-kw plumbing.deps/deps tools-deps)
        (apply-kw built-in/pom pom)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Resolve dependencies including :test alias and run Midje tests."
  [_ tools-deps OPTIONS edn "seancorfield/boot-tools-deps task."
   _ midje      OPTIONS edn "Options for zhuangxm/boot-midje task."]
  (comp (plumbing.deps/test-deps :tools-deps tools-deps)
        (plumbing.clj/midje :midje midje)))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps tools-deps
               :pom pom)
        (apply-kw built-in/jar jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (build :tools-deps tools-deps
                 :pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (build :tools-deps tools-deps
                 :pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask assemble
  "Build, add jar entries from dependencies to fileset and writes output files
   to the target directory. Mainly used for server projects which subsequently
   copy the target directory into a Docker image."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ uber       OPTIONS edn "Options for built-in/uber task."
   _ target     OPTIONS edn "Options for built-in/target task."]
  (comp (build :tools-deps tools-deps
               :pom        pom)
        (apply-kw built-in/uber uber)
        (apply-kw built-in/target target)))