(ns sock.task.porcelain.hybrid
  "Sock porcelain boot tasks for hybrid Clojure and Java projects"
  {:boot/export-tasks true}
  (:require
    [boot.core :refer [deftask]]
    [boot.task.built-in :as built-in]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.plumbing.core :refer [testing]]
    [sock.task.plumbing.clj :refer [midje]]
    [sock.task.plumbing.deps :refer [deps]]
    [sock.task.plumbing.java :refer [junit]]))

(deftask build
  "Resolve dependencies, compile Java sources, perform AOT compilation on
   Clojure namespaces, create project pom.xml file and build a jar file."
  [_ tools-deps-options OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac-options      OPTIONS edn "Options for built-in/javac task."
   _ aot-options        OPTIONS edn "Options for built-in/aot task."
   _ pom-options        OPTIONS edn "Options for built-in/pom task."
   _ jar-options        OPTIONS edn "Options for built-in/jar task."]
  (comp (apply-kw deps tools-deps-options)
        (apply-kw built-in/javac javac-options)
        (apply-kw built-in/aot aot-options)
        (apply-kw built-in/pom pom-options)
        (apply-kw built-in/jar jar-options)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)
(ns-unmap 'sock.task.porcelain.hybrid 'test)

(deftask test
  "Resolve dependencies including :test alias, compile Java sources, perform
   AOT compilation on Clojure namespaces, run Midje tests and run jUnit tests."
  [_ tools-deps-options OPTIONS edn "seancorfield/boot-tools-deps task."
   _ javac-options      OPTIONS edn "Options for built-in/javac task."
   _ aot-options        OPTIONS edn "Options for built-in/aot task."
   _ midje-options      OPTIONS edn "Options for zhuangxm/boot-midje task."
   _ junit-options      OPTIONS edn "Options for radicalzephyr/boot-junit task."]
  (comp (testing :tools-deps-options tools-deps-options)
        (apply-kw built-in/javac javac-options)
        (apply-kw built-in/aot aot)
        (midje :midje-options midje-options)
        (junit :junit-options junit-options)))

(deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps-options OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac-options      OPTIONS edn "Options for built-in/javac task."
   _ aot-options        OPTIONS edn "Options for built-in/aot task."
   _ pom-options        OPTIONS edn "Options for built-in/pom task."
   _ jar-options        OPTIONS edn "Options for built-in/jar task."
   _ install-options    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps-options tools-deps-options
               :javac-options      javac-options
               :aot-options        aot-options
               :pom-options        pom-options
               :jar-options        jar-options)
        (apply-kw built-in/install install-options)))

(deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps-options OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac-options      OPTIONS edn "Options for built-in/javac task."
   _ aot-options        OPTIONS edn "Options for built-in/aot task."
   _ pom-options        OPTIONS edn "Options for built-in/pom task."
   _ jar-options        OPTIONS edn "Options for built-in/jar task."
   _ push-options       OPTIONS edn "Options for built-in/push task."]
  (let [push-options' (merge {:repo "company-snapshots"} push-options)]
    (comp (build :tools-deps-options tools-deps-options
                 :javac-options      javac-options
                 :aot-options        aot-options
                 :pom-options        pom-options
                 :jar-options        jar-options)
          (apply-kw built-in/push push-options'))))

(deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps-options OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac-options      OPTIONS edn "Options for built-in/javac task."
   _ aot-options        OPTIONS edn "Options for built-in/aot task."
   _ pom-options        OPTIONS edn "Options for built-in/pom task."
   _ jar-options        OPTIONS edn "Options for built-in/jar task."
   _ push-options       OPTIONS edn "Options for built-in/push task."]
  (let [push-options' (merge {:repo "company-releases"} push-options)]
    (comp (build :tools-deps-options tools-deps-options
                 :javac-options      javac-options
                 :aot-options        aot-options
                 :pom-options        pom-options
                 :jar-options        jar-options)
          (apply-kw built-in/push push-options'))))
