(ns sock.task.porcelain.hybrid
  "Sock porcelain boot tasks for hybrid Clojure and Java projects"
  {:boot/export-tasks true}
  (:require
    [boot.core :refer [deftask]]
    [boot.task.built-in :as boot-built-in]
    [sock.task.plumbing.core :refer [testing]]
    [sock.task.plumbing.clj :refer [midje]]
    [sock.task.plumbing.deps :refer [deps]]
    [sock.task.plumbing.java :refer [junit]]))

(deftask build
  "Resolve dependencies, compile Java sources, perform AOT compilation on
   Clojure namespaces, create project pom.xml file and build a jar file."
  [_ javac JAVAC [str] "List of options passed to the Java compiler."
   _ aot-all  bool "Compile all Clojure namespaces."
   _ aot-namespaces NS #{sym} "The set of Clojure namespaces to compile."]
  (comp (deps)
        (boot-built-in/javac :options javac)
        (boot-built-in/aot :all aot-all
                           :namespaces aot-namespaces)
        (boot-built-in/pom)
        (boot-built-in/jar)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)
(ns-unmap 'sock.task.porcelain.hybrid 'test)

(deftask test
  "Resolve dependencies including :test alias, compile Java sources, perform
   AOT compilation on Clojure namespaces, run Midje tests and run jUnit tests."
  [_ javac JAVAC [str] "List of options passed to the Java compiler."
   _ aot-all  bool "Compile all Clojure namespaces."
   _ aot-namespaces NS #{sym} "The set of Clojure namespaces to compile."]
  (comp (testing)
        (boot-built-in/javac :options javac)
        (boot-built-in/aot :all aot-all
                           :namespaces aot-namespaces)
        (midje)
        (junit)))

(deftask local
  "Build then install project jar file to local Maven repository."
  [_ javac JAVAC [str] "List of options passed to the Java compiler."
   _ aot-all  bool "Compile all Clojure namespaces."
   _ aot-namespaces NS #{sym} "The set of Clojure namespaces to compile."]
  (comp (build :javac javac
               :aot-all aot-all
               :aot-namespaces aot-namespaces)
        (boot-built-in/install)))

(deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ javac JAVAC [str] "List of options passed to the Java compiler."
   _ aot-all  bool "Compile all Clojure namespaces."
   _ aot-namespaces NS #{sym} "The set of Clojure namespaces to compile."]
  (comp (build :javac javac
               :aot-all aot-all
               :aot-namespaces aot-namespaces)
        (boot-built-in/push :repo "company-snapshots")))

(deftask release
  "Build then deploy project jar file to company releases repository."
  [_ javac JAVAC [str] "List of options passed to the Java compiler."
   _ aot-all  bool "Compile all Clojure namespaces."
   _ aot-namespaces NS #{sym} "The set of Clojure namespaces to compile."]
  (comp (build :javac javac
               :aot-all aot-all
               :aot-namespaces aot-namespaces)
        (boot-built-in/push :repo "company-releases")))
