(ns sock.task.clj
  "Sock Boot tasks for Clojure projects."
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.deps :refer [deps]])
  (:refer-clojure :exclude [test]))

(core/deftask bat-test
  "Require and run the alternate clojure.test test runner."
  [_ opts OPTIONS edn "Options for metosin/bat-test task."]
  (core/set-env! :dependencies #(conj % '[metosin/bat-test "0.4.2" :scope "test"]))
  (require '[metosin.bat-test])
  (let [bat-test (resolve 'metosin.bat-test/bat-test)]
    (apply-kw bat-test {:cloverage true
                        :cloverage-opts {:ns-exclude-regex [#"flybot.stadium.match"]}})))

(core/deftask check []
  ;; ref: https://github.com/tolitius/boot-check
  (core/set-env! :dependencies #(conj % '[tolitius/boot-check "0.1.11" :scope "test"]))
  (require '[tolitius.boot-check])
  (let [with-yagni (resolve 'tolitius.boot-check/with-yagni)
        with-eastwood (resolve 'tolitius.boot-check/with-eastwood)
        with-kibit (resolve 'tolitius.boot-check/with-kibit)
        with-bikeshed (resolve 'tolitius.boot-check/with-bikeshed)]
    (comp
     (with-yagni)
     (with-eastwood)
     (with-kibit)
     (with-bikeshed))))

(core/deftask env-spec
  "[env] `spec`able"
  []
  (core/set-env! :dependencies #(conj % '[orchestra "2018.12.06-2" :scope "test"]))
  (core/with-pass-thru _
    (require '[orchestra.spec.test :refer [instrument]])
    ((resolve 'orchestra.spec.test/instrument))
    (require '[clojure.spec.alpha])
    ((resolve 'clojure.spec.alpha/check-asserts) true)))

(core/deftask fmt
  "Auto-format Clojure code."
  []
  (core/set-env! :dependencies #(conj % '[boot-fmt/boot-fmt "0.1.8" :scope "test"]))
  (require '[boot-fmt.core])
  (let [boot-fmt (resolve 'boot-fmt.core/fmt)]
    (boot-fmt
      :mode :overwrite
      :really true
      :source true
      :options {:style   :community
                :binding {:justify? true}
                :map     {:justify? true
                          :comma?   false}})))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Run the tests on the classpath. You probably want to compose this with the
   unit, integration and/or quickcheck tasks."
  []
  (comp (deps)
        (env-spec)
        (bat-test)))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (deps :aliases aliases)
        (apply-kw built-in/pom pom)
        (apply-kw built-in/jar jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (deps :aliases aliases)
          (apply-kw built-in/pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (deps :aliases aliases)
          (apply-kw built-in/pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask assemble
  "Build, add jar entries from dependencies to fileset and writes output files
   to the target directory. Mainly used for server projects which subsequently
   copy the target directory into a Docker image."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ uber       OPTIONS edn "Options for built-in/uber task."
   _ target     OPTIONS edn "Options for built-in/target task."]
  (comp (deps :aliases aliases)
        (apply-kw built-in/pom pom)
        (apply-kw built-in/uber uber)
        (apply-kw built-in/target target)))
