(ns sock.lib.deps-edn
  "Sock Boot functions for transformation of deps.edn to boot dependencies."
  (:require
    [boot.core :as core]
    [boot.util :as util]
    [sock.lib.io :as io]))

(defn obtain
  []
  (io/slurp-edn (io/file "deps.edn")))

(defn ->dependencies
  [deps-edn aliases]
  (->>
    ;; First merge all aliases deps into the main deps.
    (reduce
      (fn [ret k]
        (merge ret (get-in deps-edn [:aliases k :extra-deps])))
      (:deps deps-edn)
      aliases)
    ;; Second convert deps.edn data structure to boot.
    (reduce
      (fn [ret [sym {:mvn/keys [version] :keys [exclusions]}]]
        (conj! ret [sym version]))
      (transient []))
    (persistent!)))

(defn ->paths
  [deps-edn aliases]
  (->>
    (reduce
      (fn [ret k]
        (into ret (get-in deps-edn [:aliases k :extra-paths])))
      (:paths deps-edn)
      aliases)
    (into #{})))

(defn ->repositories
  [deps-edn]
  (reduce-kv #(conj %1 [%2 %3]) [] (:mvn/repos deps-edn)))


(defn repositories!
  "Merge repositories from deps.edn into the boot environment."
  [deps-edn]
  (let [wagons '[[s3-wagon-private "1.3.2"]]
        repositories (->repositories deps-edn)]
    (util/dbug* "Adding these wagons: %s\n" (util/pp-str wagons))
    (util/dbug* "Adding these repositories: %s\n" (util/pp-str repositories))
    (core/merge-env! :wagons wagons)
    (core/merge-env! :repositories repositories)))


(defn dependencies!
  "Replace boot environment dependencies with those from deps.edn."
  [deps-edn aliases]
  (let [dependencies (->dependencies deps-edn aliases)]
    (util/dbug* "Setting these dependencies: %s\n" (util/pp-str dependencies))
    (core/set-env! :dependencies dependencies)))

(defn paths!
  "Replace boot environment paths with those from deps.edn."
  [deps-edn aliases]
  (let [paths (->paths deps-edn aliases)]
    (util/dbug* "Setting these paths: %s\n" (util/pp-str paths))
    (core/set-env! :resource-paths paths)
    (core/set-env! :source-paths paths)))