(ns sock.core
  "The sock core API."
  {:boot/export-tasks true}
  (:require
    [boot.core :as boot :refer [deftask merge-env! task-options!]]
    [boot.task.built-in :refer [install jar pom push]]
    [boot.lein :as lein]
    [boot-tools-deps.core :refer [deps]]
    [clojure.edn :as edn]
    [clojure.java.io :as io]
    [clojure.set :refer [rename-keys]]
    [clojure.pprint :as pp])
  (:import [java.io File IOException FileReader PushbackReader]))

(defn- io-err
  ^IOException [fmt ^File f]
  (IOException. (format fmt (.getAbsolutePath f))))

(defn- slurp-edn-map
  "Read the file specified by the path segments, slurp it, and read it as edn."
  [^File f]
  (let [EOF (Object.)]
    (with-open [rdr (PushbackReader. (FileReader. f))]
      (let [val (edn/read {:eof EOF} rdr)]
        (cond
          (identical? val EOF) nil                          ;; empty file
          (map? val) val
          :else (throw (io-err "Expected edn map: %s" f)))))))

(deftask repositories
  "Read repositories from deps.edn and merge into the boot environment."
  [v verbose              bool  "Be verbose"]
  (let [deps (slurp-edn-map (io/file "deps.edn"))
        repositories (reduce-kv #(conj %1 [%2 %3]) [] (:mvn/repos deps))]
    (when verbose
      (println "\nAdding these repositories:")
      (pp/pprint repositories))
    (merge-env! :wagons '[[s3-wagon-private "1.2.0"]])
    (merge-env! :repositories repositories)
    identity))

(deftask pom-options
  "Set pom task options from project.edn"
  []
  (let [project (slurp-edn-map (io/file "project.edn"))]
    (task-options! pom project)
    identity))

(deftask boilerplate
  "Get this project started."
  [c config-paths    PATH [str] "the list of deps.edn files to read"
   A aliases           KW [kw]  "the list of aliases (for both -C and -R)"
   C classpath-aliases KW [kw]  "the list of classpath aliases to use"
   R resolve-aliases   KW [kw]  "the list of resolve aliases to use"
   r repeatable           bool  "Use only the specified deps.edn file for a repeatable build"
   v verbose              bool "Be verbose (and ask tools.deps to be verbose too)"]
  (comp
    (repositories
      :verbose verbose)
    (deps
      :config-paths config-paths
      :aliases aliases
      :classpath-aliases classpath-aliases
      :resolve-aliases resolve-aliases
      :repeatable repeatable
      :verbose verbose)
    (pom-options)
    (lein/generate)))


