(ns sock.core
  "The sock core API."
  {:boot/export-tasks true}
  (:require
    [boot.core :as boot :refer [deftask merge-env! task-options!]]
    [boot.task.built-in :refer [install jar pom push]]
    [boot.lein :as lein]
    [boot-tools-deps.core :refer [load-deps]]
    [clojure.edn :as edn]
    [clojure.java.io :as io])
  (:import [java.io File IOException FileReader PushbackReader]))

(defn- io-err
  ^IOException [fmt ^File f]
  (IOException. (format fmt (.getAbsolutePath f))))

(defn- slurp-edn-map
  "Read the file specified by the path segments, slurp it, and read it as edn."
  [^File f]
  (let [EOF (Object.)]
    (with-open [rdr (PushbackReader. (FileReader. f))]
      (let [val (edn/read {:eof EOF} rdr)]
        (cond
          (identical? val EOF) nil                          ;; empty file
          (map? val) val
          :else (throw (io-err "Expected edn map: %s" f)))))))

(defn repositories! []
  (let [deps (slurp-edn-map (io/file "deps.edn"))
        repositories (:mvn/repos deps)]
    (pr repositories)
    (merge-env! :wagons '[[s3-wagon-private "1.2.0"]])
    (merge-env! :repositories repositories)))

(defn pom! []
  (let [project (slurp-edn-map (io/file "project.edn"))]
    (task-options! pom project)))

(defn setup! []
  (lein/generate)
  (repositories!)
  (load-deps)
  (pom!))

