;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: David Creemer
;;
;; flipboard.component.memcache
;;
;; manages memcache client using flipboard settings

(ns flipboard.component.memcache
  (:require [flipboard.component.dynconfig :as dc]
            [flipboard.base.memcache :as mc]
            [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]))

(defn- get-prefix
  [comp]
  (if-let [p (:prefix comp)]
    (str p ".")))

(defn- new-connection
  "create a new memcached connection using config on the component."
  [comp]
  (let [dynconfig (:dynconfig comp)
        file (:dynconfig-file comp)
        cfg (get-in comp [:core :config])
        hosts-key (str (get-prefix comp)
                       "memcached."
                       (:fl-env cfg) "."
                       (:fl-pool cfg) ".hosts")
        timeout-key (str (get-prefix comp) "memcached.timeout")
        timeout (dc/get-int-val dynconfig file timeout-key)
        servers (dc/get-str-val dynconfig file hosts-key)]
    (log/info "memcache host config" hosts-key " connecting to " servers)
    (mc/connect servers timeout)))

(defn- update-connection
  [comp]
  (->> (new-connection comp)
       (reset! (:connection comp))))

;
; public interface
;
(defn get-client
  "returns the memcache client in this component"
  [comp]
  (deref (:connection comp)))

(defn get-val
  "convenient method to get a value with key"
  [comp key]
  (-> (get-client comp)
      (.get key)))

(defn delete-val
  "convenient method to delete a key"
  [comp key]
  (-> (get-client comp)
      (.delete key)))

(defn set-val
  "convenient method to set a value on key with timeout
  expires in seconds.
  returns val for threading convenience"
  [comp key expires val]
  (if (nil? val)
    (delete-val comp key)
    (-> (get-client comp)
        (.set key expires val)))
  val)

; component related
(defrecord MemcachedConnection [dynconfig prefix]
  component/Lifecycle

  (start [this]
    (let [file "memcache.config"
          ; must first init the connection so it can be updated
          newme (assoc this :connection (atom nil)
                       :dynconfig-file file)
          dynconfig (:dynconfig this)
          ; must create handler on using newme
          ; so :connection atom is set
          handler #(update-connection newme)]
      (dc/watch-properties dynconfig file handler)
      (log/info "started memcache component")
      newme))

  (stop [this]
    (.shutdown (get-client this))
    (log/info "stopped memcache component")
    (assoc this :connection nil)))

(defn new-memcached
  "create a new memcached component with
  dynconfig memcache.config property names
  with optional memcache prefix using:
  <prefix>.memcached.<env>.<pool>.hosts
  <prefix>.memcached.timeout"
  [& [prefix]]
  (map->MemcachedConnection {:prefix prefix}))
