;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: Howard Zhao
;;
;; flipboard.component.dynconfig
;; a component that can be asked to watch files
;; and call cooresponding handler
;; For property based dynconfig, default
;; implementation of handler is reload the config
(ns flipboard.component.dynconfig
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [flipboard.base.watchtower :as wt]
            [flipboard.base.config :as config]
            [flipboard.base.util :refer [to-int]]
            [clojurewerkz.propertied.properties :as props]
            [clojure.java.io :as io]))

(defn- set-file-prop
  "set the property map for a given dynconfig file, most useful for testing"
  [dyncomp file new-prop]
  (-> (:prop-map dyncomp)
      (swap! assoc file new-prop)))

(defn- load-single-file
  "load a single dynconfig file used in watcher handler.
  file is the watched file name also used as key in the watcher.
  changed-file is the file actually changed."
  [dyncomp file changed-file]
  (log/info "loading changed dynconfig" changed-file)
  (let [fileobj (if (instance? String changed-file)
                  (io/file changed-file)
                  changed-file)
        new-prop (-> fileobj
                     (props/load-from)
                     (props/properties->map))]
    (set-file-prop dyncomp file new-prop)
    (log/info "Loaded changed dynconfig" file)))

;;
;; public interface for Dynconfig component
;;
(defn watch-file
  "watch dynconfig file and call func on change
  func should take a file object and do useful side effects."
  [dyncomp file func]
  (let [watches (:watches dyncomp)
        cfg (get-in dyncomp [:core :config])
        fullpath (config/dynpath cfg file)
        w (wt/watcher
            fullpath
            (wt/rate 5000) ; every 5 sec
            (wt/on-change
              (fn [fobj] (func fullpath))))]
    ; load for first time
    (func fullpath)
    (log/info "start watching" fullpath)
    (swap! watches assoc file w)))

(defn watch-properties
  "watch dynconfig file and if changed, load it as properties.
  if func is provided, call the func after updating properties."
  [dyncomp file & [func]]
  (letfn [(handler
            [changed-file]
            (load-single-file dyncomp file changed-file)
            (if func (func)))]
    (watch-file dyncomp file handler)))

(defn get-configs
  "get all properties: map from file -> property map"
  [dyncomp]
  (deref (:prop-map dyncomp)))

(defn get-str-val
  "get the dynconfig str value of a given dynconfig file and key"
  [dyncomp file key & [default]]
  (-> (get-configs dyncomp)
      (get-in [file key])
      (or default)))

(defn get-int-val
  "get the dynconfig int value of a given dynconfig file and key"
  [dyncomp file key & [default]]
  (-> (get-str-val dyncomp file key)
      (or default)
      (to-int)))

(defrecord Dynconfig [core]
  component/Lifecycle

  (start [this]
    (log/info "starting dynconfig")
    (assoc this
           ; maps a list of fullpathes to corresponding watcher
           :watches (atom {})
           ; maps dynconfig to value
           :prop-map (atom {})))

  (stop [this]
    (log/info "stopping dynconfig")
    (doseq [watch (-> (:watches this)
                      (deref)
                      (vals))]
      (future-cancel watch))
    (assoc this :watches nil :prop-map nil)))

(defn new-dynconfig
  "create a new dynconfig component. "
  []
  (map->Dynconfig {}))
