;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: David Creemer
;;
;; flipboard.base.kafka
;;
;; thin wrapper on Kafka consumer code, parts borrowed from
;; https://github.com/pingles/clj-kafka

(ns flipboard.base.kafka
  (:require [clojure.core.async :as async])
  (:import [java.util Properties]
           [kafka.javaapi.consumer ConsumerConnector]
           [kafka.consumer ConsumerConfig Consumer KafkaStream ConsumerIterator]))

(defrecord KafkaMessage [topic offset partition key value])

(defn- as-properties
  [m]
  (let [props (Properties. )]
    (doseq [[n v] m] (.setProperty props n v))
    props))

(defn- lazy-iterate
  [it]
  (lazy-seq
   (when (.hasNext it)
     (cons (.next it) (lazy-iterate it)))))

(defn- to-message
  "convert a MessageAndMetadata to KafkaMessage record"
  [m]
  (KafkaMessage. (.topic m) (.offset m) (.partition m) (.key m) (.message m)))

;; low-level api
(defn consumer
  [m]
  (let [config (ConsumerConfig. (as-properties m))]
    (Consumer/createJavaConsumerConnector config)))

(defn shutdown
  "Closes the connection to Zookeeper and stops consuming messages."
  [^ConsumerConnector consumer]
  (.shutdown consumer))

;; stream api

(defn messages
  "return a lazy sequence of nice KafkaMessage records from a stream iterator"
  [^ConsumerIterator it]
  (map to-message (lazy-iterate it)))

(defn create-stream
  "create an iterator for a message stream for a single topic"
  [^ConsumerConnector consumer topic & {:keys [threads]
                                        :or   {threads 1}}]
  (let [[_topic [stream & _]] (first (.createMessageStreams consumer {topic (int threads)}))]
    (.iterator ^KafkaStream stream)))

;; core.async api

(defn stream->channel
  "given a kafka message stream and a core.async channel,
  put the message value on the channel, possible transformed by xform"
  [stream chan & {:keys [xform] :or {xform identity}}]
  (if (.hasNext stream)
    (async/thread
      (loop [m (.next stream)]
        (async/put! chan (xform (to-message m)))
        (if (.hasNext stream)
          (recur (.next stream)))))))
