;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: David Creemer
;;
;; flipboard.base.domains
;;
;; domains.yml handling
;;
;; TODO:
;; process types type:
;; handle 'string("xxx")' case
;; convert to date and duration
;; understand about merge vs. replace for parameters (e.g. flus.removeQueryParams)
;; - most use replace [just pick first found], others may use set-union
;; validate

(ns flipboard.base.domains
  (:require [clojure.string :as str]
            [clojure.instant :as inst]
            [schema.core :as s]
            [clj-yaml.core :as yaml]))

(defonce scheme-prefix-re (re-pattern "^http[s]?://"))
(defonce path-re (re-pattern "^/*(.*)"))


(defn- cleanup
  "cleanup some common errors in domain/path specification"
  [d]
  (-> d
      (str/lower-case)
      (str/replace-first scheme-prefix-re "")))

(defn- parse-domain-path
  "given a map with domain/path (e.g. example.com/foo/bar or just example.com), split it into
  the domain part and the path regular expression part, and replace :domain with the cleaned
  domain, and add :path-re with the path regex"
  [m]
  (let [[dom path] (-> (:domain m)
                       (cleanup)
                       (str/split #"/" 2))
        path       (if (str/blank? path)
                     "^/"
                     (str/replace path path-re "^/$1"))]
    (assoc m :domain dom :path-re (re-pattern path))))


(defn- remove-rules
  "remove all the '*rules' attributes"
  [m]
  (let [rules-keys (filter #(.endsWith (name %) "rules") (keys m))]
    (apply dissoc m rules-keys)))


(defn- coerce-to-type
  "ensure that v is or converts to the given type (this cleans up after the YAML parsing, which
  does much of the type coercion)."
  [k v type]
  (cond
   (= type s/Regex)          [k (re-pattern v)]
   (= type [s/Regex])        [k (map re-pattern v)]
   (= type #{s/Str})         [k (set v)]
   :else                     [k v]))


(defn- merge-vals
  "given an ordered list of values, merge them according to the type"
  [k type vals]
  (cond
   (= type #{s/Str}) (apply clojure.set/union vals)
   :else (first vals)))

(defn- xform-entries
  "potentially transform all the k-v entries in a domain map"
  [types m]
  (let [xform (fn [[k v]]
                (let [type (types k)]
                  (coerce-to-type k v type)))]
    (into {} (map xform m))))


(defn- read-domains
  "read domain rules from yaml, and check/xform against types map"
  [y types]
  (let [rules (->> (yaml/parse-string y)
                   (filter #(contains? % :domain))       ; only consider domains
                   (map remove-rules)                    ; remove all *rules keys
                   (map (partial xform-entries types))   ; validate and transform k-v to k'v' according to types
                   (map parse-domain-path)               ; extract the path-re
                   (group-by :domain))]                  ; convert to map of list of maps
    (assoc rules :types types)))


(defn- type-to-schema
  [tp]
  (case tp
    "boolean" s/Bool
    "date" s/Inst
    "float" s/Num
    "string" s/Str
    "int" s/Int
    "regex" s/Regex
    "vector of string" [s/Str]
    "set of string" #{s/Str}
    "vector of int" [s/Int]
    "vector of regex" [s/Regex]
    s/Str))


(defn- read-types
  [t]
  (let [hacks {:flus.removeFragments "set of string" ; currently not correctly set in types file.
               :flus.removeQueryParams "set of string"
               :flus.keepQueryParams "set of string"}
        parse-var (fn [acc m]
                    (let [varname (keyword (:variable m))
                          type (get hacks varname (:type m))
                          type (type-to-schema type)]
                      (assoc acc varname type)))]
    (reduce parse-var {} (yaml/parse-string t))))


;; to find a var, lookup the domain from most to least significant. For each domain,
;; scan the rules from last to first to determine if the URL regex matches. If it
;; does, then see if the variable exists at that level


(defn- enumerate-parts
  "given a domain, return a seq of the composed parts of the domain. e.g.
  www.example.com -> ('*' 'com' 'example.com' 'www.example.com')"
  [host]
  (let [parts (reverse (str/split host #"\."))              ; www.google.com -> ("com" "google" "www")
        parts (rest (reductions conj [] parts))]    ; (["com"] ["com" "google"] ["com" "google" "www"])
    (into ["*"] (map #(str/join "." (reverse %)) parts))))


(defn- host-path-gig-maps
  "return a lazy list of rule maps for the given host that either match the given gig, or have
  no gig (and therefore apply to all lookups)"
  [rules host path gig]
  (let [maplist (reverse (get rules host []))
        gigs    (set [gig ""])]
    (->> maplist
         (filter #(re-find (:path-re %) path)) ; find all maps for the host that match the path
         (filter #(gigs (get % :gig ""))))))   ; and that match the gig


(defn- get-maps
  "return a list of all maps matching the host components, path, and gig from the rules"
  [rules host path gig]
  (let [hosts (reverse (enumerate-parts host))]
    (->> hosts                                  ; for each host componenet
         (map #(host-path-gig-maps rules % path gig)) ; generate a list of maps
         (flatten)                              ; flatten it
         (filter not-empty))))                  ; remove empties


;; todo: walk the list of maps, applying the select function to either pick first, union sets,
;; append to list, etc. as per the :key rules (and eventually the type...)
(defn- lookup-vals
  "return a merged map of all maps matching the host components, path, and gig"
  [rules host path k & {:keys [gig] :or {gig ""}}]
  (->> (get-maps rules host path gig) ; get all of the maps
       (map #(get % k))               ; convert list of maps to list of values (or nil)
       (filter some?)))               ; remove empties


;;
;; public interface
;;


(defn load-domains-yaml
  "load the domains.yml file from the given path into a rules map."
  [domains-path types-path]
  (let [types (read-types (slurp types-path))]
    (read-domains (slurp domains-path) types)))


(defn lookup-key
  "given a host, path, key, and optional gig, return the keys value in the map, or the :default
  value if provided"
  [rules host path k & {:keys [gig default] :or {gig ""}}]
  (let [vals (lookup-vals rules host path k :gig gig)
        types (:types rules)]
    (or (merge-vals k (types k) vals) default)))
