# Collage

Collage is a simple-to-use image processing library for Clojure. It's intended
to be a drop-in solution for high-level image processing needs. It draws some
inspiration from Python's [PIL](http://effbot.org/imagingbook/pil-index.htm) and
is somewhat similar to mikera's [imagez](https://github.com/mikera/imagez).

### Motivation

Collage grew out of my own need when I was writing another Clojure application.
I wanted to have some high-level image processing functionality available -
functions into which I could just pass some `BufferedImages` and get back
transformed `BufferedImages`. At the time, I resorted to doing Java interop,
which is nice, but Clojure is nicer. I also wanted to get more experience with
Clojure in general.

### Project goals

- Ease of use
- Well tested-ness
- Clean, composable internal API
- Learn (more idiomatic) Clojure

Documentation generated by Marginalia is available on
[Github pages](https://karls.github.io/collage/).

Available on [Clojars](https://clojars.org/fivetonine/collage).

## Usage

#### Using the `with-image` macro.

```clj
(:require [fivetonine.collage.util :as util])
(:require [fivetonine.collage.core :refer :all])

(with-image "/path/to/image.jpg"
            (resize :width 1000)
            (rotate 90)
            (util/save :quality 0.85))
```

Loads the image at `/path/to/image.jpg`, resizes it to have width of 1000px
(height is computed automatically), rotates 90 degrees clockwise and saves it
with 85% quality of the original, overwriting the original.

```clj
(:require [fivetonine.collage.util :as util])
(:require [fivetonine.collage.core :refer :all])

(def image (util/load-image "/path/to/image.jpg"))
(with-image image
	        (crop 100 50 200 100)
	        (util/save "/path/to/new-image.jpg" :quality 1.0))
```

Loads an image at `/path/to/image.jpg`. With that image, crops a 200px by 100px
image out of the original, at the point (100,50) in the original image, saves
it with 100% quality at `/path/to/new-image.jpg`.

#### Vanilla functions.

Plain ol' functions

```clj
(:require [fivetonine.collage.util :as util])
(:require [fivetonine.collage.core :refer :all])

(def image (load-image "/path/to/image.png"))
(util/save (flip image :horizontal))
```

or using the [thread-first macro](http://clojuredocs.org/clojure_core/clojure.core/-%3E).

```clj
(:require [fivetonine.collage.util :as util])
(:require [fivetonine.collage.core :refer :all])

(-> (load-image "/path/to/image.png") (flip :horizontal) (util/save :progressive true))
```

## WebP support

Collage does not provide support for WebP out of the box. Collage includes all
the JVM `ImageIO` API plumbing (in resources/webp-imageio.jar), but
the native binaries are not provided. Note that the `webp` format is reported as
a supported format with `ImageIO.getReaderFormatNames()`. But when trying
to load a `.webp` image, an exception is thrown as the native binary for
actually loading the raw data in is missing.

The main reason for not providing binaries is that I don't want to
build and maintain all the versions of all the binaries for all the platforms.

Compiling the binary is fairly straightforward. Some instructions are available
in the luciad's [webp-imageio](https://bitbucket.org/luciad/webp-imageio) repo.
This repo includes code that is used in Collage to provide support for WebP.
I also compiled the native binary that WebP depends on using the instructions
in the same repo.

If you run into problems compiling the binary or getting all the pieces working
together, please open an issue and describe the situation.

WebP support is currently tested only on Mac OSX, but it _should_ work on other
platforms as well.

## Contributing

Contributions, suggestions and friendly criticism are all welcome.

## License

Copyright © 2013 Karl Sutt

Distributed under the Eclipse Public License either version 1.0.
