
// Detect Hardware
# define FERRET_CONFIG_SAFE_MODE TRUE

#if !defined(FERRET_SAFE_MODE)
  #if defined(__APPLE__) ||                       \
    defined(_WIN32) ||                            \
    defined(__linux__) ||                         \
    defined(__unix__) ||                          \
    defined(_POSIX_VERSION)
  
    # undef  FERRET_CONFIG_SAFE_MODE
    # define FERRET_STD_LIB TRUE
  #endif
  
  #if defined(ARDUINO)

    # define FERRET_HARDWARE_ARDUINO TRUE

    #if !defined(FERRET_HARDWARE_ARDUINO_UART_PORT)
      # define FERRET_HARDWARE_ARDUINO_UART_PORT Serial
    #endif
  #endif
  
  #if defined(FERRET_HARDWARE_ARDUINO)
    # undef  FERRET_CONFIG_SAFE_MODE
    # define FERRET_DISABLE_STD_MAIN TRUE


    #if defined(__AVR__)
      # undef  FERRET_MEMORY_POOL_PAGE_TYPE
      # define FERRET_MEMORY_POOL_PAGE_TYPE byte
    #endif

  #endif
#endif

#if defined(FERRET_CONFIG_SAFE_MODE)
  # define FERRET_DISABLE_MULTI_THREADING TRUE
  # define FERRET_DISABLE_STD_OUT TRUE
#endif
#ifdef FERRET_STD_LIB
 #include <iostream>
 #include <iomanip>
 #include <sstream>
 #include <cstdio>
 #include <cstdlib>
 #include <cstddef>
 #include <cmath>
 #include <vector>
 #include <algorithm>
 #include <chrono>
 #include <atomic>
 #include <mutex>
 #include <thread>
 #include <future>
#endif

#ifdef FERRET_HARDWARE_ARDUINO
 #include <Arduino.h>
 #include <stdio.h>
 #include <stdlib.h>
 #include <stdint.h>
#endif

#ifdef FERRET_CONFIG_SAFE_MODE
 #include <stdio.h>
 #include <stdlib.h>
 #include <stdint.h>
 #include <math.h>
#endif

namespace ferret{
  // Types
  typedef uint8_t byte;
  #pragma pack(push, 1)
  class int24_t {
  protected:
    byte word[3];
  public:
    int24_t(){ }
  
    template<typename T>
    explicit int24_t( T const & val ) {
      *this   = (int32_t)val;
    }
  
    int24_t( const int24_t& val ) {
      *this   = val;
    }
  
    operator int32_t() const {
      if (word[2] & 0x80) { // negative? - then sign extend.
        return
          (int32_t)(((uint32_t)0xff    << 24) |
                    ((uint32_t)word[2] << 16) |
                    ((uint32_t)word[1] << 8)  |
                    ((uint32_t)word[0] << 0));
      }else{
        return
          (int32_t)(((uint32_t)word[2] << 16) |
                    ((uint32_t)word[1] << 8)  |
                    ((uint32_t)word[0] << 0));
      }
    }
  
    int24_t& operator =( const int24_t& input ) {
      word[0]   = input.word[0];
      word[1]   = input.word[1];
      word[2]   = input.word[2];
  
      return *this;
    }
  
    int24_t& operator =( const int32_t input ) {
      word[0]   = ((byte*)&input)[0];
      word[1]   = ((byte*)&input)[1];
      word[2]   = ((byte*)&input)[2];
  
      return *this;
    }
  
    int24_t operator +( const int24_t& val ) const {
      return int24_t( (int32_t)*this + (int32_t)val );
    }
  
    int24_t operator -( const int24_t& val ) const {
      return int24_t( (int32_t)*this - (int32_t)val );
    }
  
    int24_t operator *( const int24_t& val ) const {
      return int24_t( (int32_t)*this * (int32_t)val );
    }
  
    int24_t operator /( const int24_t& val ) const {
      return int24_t( (int32_t)*this / (int32_t)val );
    }
  
    int24_t& operator +=( const int24_t& val ) {
      *this   = *this + val;
      return *this;
    }
  
    int24_t& operator -=( const int24_t& val ) {
      *this   = *this - val;
      return *this;
    }
  
    int24_t& operator *=( const int24_t& val ) {
      *this   = *this * val;
      return *this;
    }
  
    int24_t& operator /=( const int24_t& val ) {
      *this   = *this / val;
      return *this;
    }
  
    int24_t operator -() {
      return int24_t( -(int32_t)*this );
    }
  
    bool operator ==( const int24_t& val ) const {
      return (int32_t)*this == (int32_t)val;
    }
  
    bool operator !=( const int24_t& val ) const {
      return (int32_t)*this != (int32_t)val;
    }
  
    bool operator >=( const int24_t& val ) const {
      return (int32_t)*this >= (int32_t)val;
    }
  
    bool operator <=( const int24_t& val ) const {
      return (int32_t)*this <= (int32_t)val;
    }
  
    bool operator >( const int24_t& val ) const {
      return (int32_t)*this > (int32_t)val;
    }
  
    bool operator <( const int24_t& val ) const {
      return (int32_t)*this < (int32_t)val;
    }
  };
  #pragma pack(pop)
  // Concurrency
  #if defined(FERRET_DISABLE_MULTI_THREADING)
    class mutex {
    public:
      void lock()   {} 
      void unlock() {} 
    };
  #else
    #if defined(FERRET_STD_LIB)
      class mutex {
        ::std::mutex m;
      public:
        void lock()   { m.lock();   } 
        void unlock() { m.unlock(); }
      };
    #endif
  
    #if defined(FERRET_HARDWARE_ARDUINO)
      class mutex {
      public:
        void lock()   { noInterrupts(); } 
        void unlock() { interrupts();   }
      };
    #endif
  #endif
  
  class lock_guard{
    mutex & _ref;
  public:
    explicit lock_guard(const lock_guard &) = delete;
    explicit lock_guard(mutex & mutex) : _ref(mutex) { _ref.lock(); };
    ~lock_guard() { _ref.unlock(); }
  };
  // Containers
  template<class Type>
  struct array_seq {
    Type* p;
    explicit array_seq(Type* p) : p(p) {}
    bool operator!=(array_seq rhs) {return p != rhs.p;}
    Type& operator*() {return *p;}
    void operator++() {++p;}
  };
   
  template<class T, size_t S>
  struct array {
    T data[S];
        
    T& operator [](int idx)      { return data[idx]; }
    T operator [](int idx) const { return data[idx]; }
      
    array_seq<T> begin() { return array_seq<T>(data);      }
    array_seq<T> end()   { return array_seq<T>(data + S);  }
    size_t       size()  { return S;                       }
  };
  template<size_t S>
  class bitset {
  private:
    byte bits[S / 8 + 1];
             
    inline size_t index (size_t i) { return i / 8; }
    inline size_t offset(size_t i) { return i % 8; }
               
  public:
               
    bitset() : bits{ (byte)0x00 } { }
             
    inline void set   (size_t b){
      bits[index(b)] = (byte)((int)bits[index(b)] |  (1 << (offset(b))));
    }
               
    inline void reset (size_t b){
      bits[index(b)] = (byte)((int)bits[index(b)] & ~(1 << (offset(b))));
    }
               
    inline bool test  (size_t b){
      return ((int)bits[index(b)] & (1 << (offset(b))));
    }
  };
}

// Math
namespace ferret{
  constexpr auto operator "" _MB( unsigned long long const x ) -> long {
    return 1024L * 1024L * (long)x;
  }
    
  constexpr auto operator "" _KB( unsigned long long const x ) -> long {
    return 1024L * (long)x;
  }
    
  constexpr auto operator "" _pi(long double x) -> double {
    return 3.14159265358979323846 * (double)x;
  }
  
  constexpr auto operator "" _pi(unsigned long long int  x) -> double {
    return 1.0_pi * (double)x;
  }
  
  constexpr auto operator "" _deg(long double x) -> double {
    return (1.0_pi * (double)x) / 180;
  }
  
  constexpr auto operator "" _deg(unsigned long long int  x) -> double {
    return 1.0_deg * (double)x;
  }
  #if !defined(__clang__)
  constexpr auto operator "" _QN(long double x) -> int {
    return (int)::floor(::log(1.0/(double)x)/::log(2));
  }
  #endif
  
  template<int bits> struct fixed_real_container;
  template<> struct fixed_real_container<8>  { typedef int8_t  base_type;
                                               typedef int16_t next_type; };
  template<> struct fixed_real_container<16> { typedef int16_t base_type;
                                               typedef int24_t next_type; };
  template<> struct fixed_real_container<24> { typedef int24_t base_type;
                                               typedef int32_t next_type; };
  template<> struct fixed_real_container<32> { typedef int32_t base_type;
                                               typedef int64_t next_type; };
  template<> struct fixed_real_container<64> { typedef int64_t base_type;
                                               typedef int64_t next_type; };
  
  #pragma pack(push, 1)
  template<int bits, int exp>
  class fixed_real {
    typedef fixed_real fixed;
    typedef typename fixed_real_container<bits>::base_type base;
    typedef typename fixed_real_container<bits>::next_type next;
  
    base m;
    static const int N      = (exp - 1);
    static const int factor = 1 << N;
  
    template<typename T>
    inline base from(T d) const { return (base)(d * factor); }
  
    template<typename T>
    inline T to_rational() const { return T(m) / factor; }
  
    template<typename T>
    inline T to_whole() const { return (T)(m >> N); }
      
  public:
  
    //from types
    explicit fixed_real( )           : m(0) { }
    template<typename T>
    explicit fixed_real(T v)         : m(from<T>(v)) {}
  
    template<typename T>
    fixed& operator=(T v)        { m = from<T>(v); return *this; }
      
    //to types
    template<typename T>
    operator T()           const { return to_whole<T>();    }
    operator double()      const { return to_rational<double>(); }
      
    // operations
    fixed& operator+= (const fixed& x) { m += x.m; return *this; }
    fixed& operator-= (const fixed& x) { m -= x.m; return *this; }
    fixed& operator*= (const fixed& x) { m = (base)(((next)m * (next)x.m) >> N); return *this; }
    fixed& operator/= (const fixed& x) { m = (base)(((next)m * factor) / x.m); return *this; }
    fixed& operator*= (int x)          { m *= x; return *this; }
    fixed& operator/= (int x)          { m /= x; return *this; }
    fixed  operator-  ( )              { return fixed(-m); }
      
    // friend functions
    friend fixed operator+ (fixed x, const fixed& y) { return x += y; }
    friend fixed operator- (fixed x, const fixed& y) { return x -= y; }
    friend fixed operator* (fixed x, const fixed& y) { return x *= y; }
    friend fixed operator/ (fixed x, const fixed& y) { return x /= y; }
      
    // comparison operators
    friend bool operator== (const fixed& x, const fixed& y) { return x.m == y.m; }
    friend bool operator!= (const fixed& x, const fixed& y) { return x.m != y.m; }
    friend bool operator>  (const fixed& x, const fixed& y) { return x.m > y.m; }
    friend bool operator<  (const fixed& x, const fixed& y) { return x.m < y.m; }
    friend bool operator>= (const fixed& x, const fixed& y) { return x.m >= y.m; }
    friend bool operator<= (const fixed& x, const fixed& y) { return x.m <= y.m; }
  
  #if defined(FERRET_STD_LIB)
    friend std::ostream& operator<< (std::ostream& stream, const fixed& x) {
      stream << (double)x;
      return stream;
    }
  #endif
  };
  #pragma pack(pop)
  #if !defined(FERRET_NUMBER_TYPE)
     #define FERRET_NUMBER_TYPE int
  #endif
  
  #if !defined(FERRET_REAL_TYPE)
     #define FERRET_REAL_TYPE   double
  #endif
  
  #if !defined(FERRET_REAL_EPSILON)
     #define FERRET_REAL_EPSILON   0.00001
  #endif
    
    typedef FERRET_NUMBER_TYPE           number_t;                   // Whole number Container.
    typedef FERRET_REAL_TYPE             real_t;                     // Real number Container.
    const   real_t                       real_epsilon(FERRET_REAL_EPSILON);
  #if !defined(FERRET_DISABLE_STD_OUT)
    const   size_t                       number_precision = 4;       // number Format String (fprinf)
  #endif
  namespace runtime{
    #undef min
    #undef max
    #undef abs
  
    template <typename T>
    static constexpr T max(T a, T b) {
      return a < b ? b : a;
    }
  
    template <typename T, typename... Ts>
    static constexpr T max(T a, Ts... bs) {
      return max(a, max(bs...));
    }
    
    template<typename T>
    constexpr T min(T a, T b){
      return ((a) < (b) ? (a) : (b));
    }
  
    template <typename T, typename... Ts>
    static constexpr T min(T a, Ts... bs) {
      return min(a, min(bs...));
    }
  
    template<typename T>
    constexpr T abs(T a){
      return ((a) < (T)0 ? -(a) : (a));
    }
  }
  namespace euclidean{
    template <size_t D, typename T>
    struct vector {
      T d[D];
   
      T& operator [](size_t idx)      { return d[idx]; }
      T operator [](size_t idx) const { return d[idx]; }
      
      friend vector operator+ (vector u, const vector& v) {
        vector result;
        for (size_t i = 0; i < D; i++)
          result.d[i] = u.d[i] + v.d[i];
        return result;
      }
   
      friend vector operator- (vector u, const vector& v) {
        vector result;
        for (size_t i = 0; i < D; i++)
          result.d[i] = u.d[i] - v.d[i];
        return result;
      }
   
      friend vector operator* (vector u, const T& v) {
        vector result;
        for (size_t i = 0; i < D; i++)
          result.d[i] = u.d[i] * v;
        return result;
      }
   
      friend bool operator== (const vector& x, const vector& y) {
        for (size_t i = 0; i < D; i++)
          if (x.d[i] != y.d[i])
            return false;
        return true;
      }
   
      T magnitude(){
        T acc = 0;
        for(size_t i = 0; i < D; i++){
          T t = d[i] * d[i];
          acc += t;
        }
   
        return sqrt(acc);
      }
   
      vector normalize(){
        T mag = magnitude();
        if (mag == 0)
          return vector{{0}};
   
        vector r;
        for(size_t i = 0; i < D; i++){
          r[i] = d[i] / mag;
        }
        
        return r;
      }
   
      T dist(vector v){ return ((*this) - v).magnitude(); }
   
  #if defined(AUTOMATON_STD_LIB)
      friend std::ostream& operator<< (std::ostream& stream, const vector& x) {
        stream << '[';
        for (size_t i = 0; i < D - 1; i++)
          stream << x.d[i] << ',';
        stream << x.d[D - 1] << ']';
        return stream;
      }
  #endif
    };
  
    typedef euclidean::vector<2, real_t> vector_2d;
    typedef euclidean::vector<3, real_t> vector_3d;
  }
}

// Initialize Hardware
namespace ferret{
  #if !defined(FERRET_UART_RATE)
    # define FERRET_UART_RATE 9600
  #endif
  #if !defined(FERRET_IO_STREAM_SIZE)
    # define FERRET_IO_STREAM_SIZE 80
  #endif
  #if defined(FERRET_DISABLE_STD_OUT)
     namespace runtime{
       void init(){ }
      
       template <typename T>
       void print(T){ }
     }
  #endif
  #if defined(FERRET_STD_LIB) && !defined(FERRET_DISABLE_STD_OUT)
    namespace runtime{
      void init(){}
      
      template <typename T>
      void print(const T & t){ std::cout << t; }
  
      template <>
      void print(const real_t & n){
        std::cout << std::fixed << std::setprecision(number_precision) << n;
      }
  
      void read_line(char *buff, std::streamsize len){
        std::cin.getline(buff, len);
      }
    }
  #endif
  #if defined(FERRET_HARDWARE_ARDUINO) && !defined(FERRET_DISABLE_STD_OUT) 
    namespace runtime{
      void init(){ FERRET_HARDWARE_ARDUINO_UART_PORT.begin(FERRET_UART_RATE); }
  
      template <typename T>
      void print(const T t){ FERRET_HARDWARE_ARDUINO_UART_PORT.print(t); }
  
      template <>
      // cppcheck-suppress passedByValue
      void print(const real_t d){ FERRET_HARDWARE_ARDUINO_UART_PORT.print(double(d)); }
      
      template <>
      void print(void *p){
        FERRET_HARDWARE_ARDUINO_UART_PORT.print((size_t)p,HEX);
      }
  
      template <> void print(const void * const p){
        FERRET_HARDWARE_ARDUINO_UART_PORT.print((size_t)p, HEX);
      }
  
      void read_line(char *buff, size_t len){
        byte idx = 0;
        char c;
        do{
          while (FERRET_HARDWARE_ARDUINO_UART_PORT.available() == 0);
          c = FERRET_HARDWARE_ARDUINO_UART_PORT.read();
          buff[idx++] = c;
        }while (c != '\n');
        buff[--idx] = 0x00;
      }
     }
  #endif
}

// Object System Base
namespace ferret{
  namespace memory {
    template <typename T>
    class pointer{
      T *ptr;
  
    public:
  
      inline explicit pointer(T *p = nullptr) : ptr(p){ }
      inline operator T* () const { return ptr; }
  
      inline pointer& operator= (T *other){
        ptr = other;
        return *this;
      }
  
      inline T *operator->() const { return ptr; }
    };
  }
  namespace memory{
    inline size_t align_of(uintptr_t size, size_t align){
      return (size + align - 1) & ~(align - 1);
    }
  
    template<class T>
    size_t align_of(const void * ptr) {
      return align_of(reinterpret_cast<uintptr_t>(ptr), sizeof(T));
    }
      
    inline size_t align_req(uintptr_t size, size_t align){
      size_t adjust = align - (size & (align - 1));
        
      if(adjust == align)
        return 0;
      return adjust;
    }
  
    template<class T>
    size_t align_req(const void * ptr) {
      return align_req(reinterpret_cast<uintptr_t>(ptr), sizeof(T));
    }
  
    template <typename... Ts>
    constexpr size_t max_sizeof() {
      return runtime::max(sizeof(Ts)...);
    }
  }
  #ifdef FERRET_MEMORY_POOL_SIZE
  namespace memory{
    namespace allocator{
      template<typename page_size, size_t pool_size>
      class memory_pool{
      public:
        bitset<pool_size> used;
        page_size pool[pool_size];
        size_t offset;
        size_t page_not_found;
  
        memory_pool() : pool{0}, offset(0), page_not_found(pool_size + 1) { }
  
        inline size_t chunk_length(size_t size){
          size_t d = (size / sizeof(page_size));
          size_t f = (size % sizeof(page_size));
  
          if (f == 0)
            return d;
          else
            return (d + 1);
        }
  
        inline bool chunk_usable(size_t begin, size_t end){
          for(size_t i=begin; i < end; i++)
            if (used.test(i))
              return false;
          return true;
        }
  
        inline size_t next_page(size_t begin){
          for(size_t i=begin; i < pool_size; i++)
            if (!used.test(i))
              return i;
          return pool_size;
        }
  
        inline size_t scan_pool(size_t pages_needed, size_t offset = 0){
          for(;;){
            size_t begin = next_page(offset);
            size_t end   = begin + pages_needed;
    
            if (end > pool_size)
              return page_not_found;
          
            if (chunk_usable(begin, end))
              return begin;
    
            offset = end;
          }
        }
  
        void *allocate(size_t req_size){
          size_t length = chunk_length(++req_size);
          size_t page   = scan_pool(length, offset);
  
          if (page == page_not_found){
            page = scan_pool(length);
            if (page == page_not_found)
              return nullptr;
          }
          
          pool[page] = length;
          offset = page + length;
          for(size_t i = page; i < offset; i++)
            used.set(i);
  
          return &pool[++page];
        }
  
        void free(void *p){
          ptrdiff_t begin = (static_cast<page_size *>(p) - pool) - 1;
          ptrdiff_t end = begin + (ptrdiff_t)pool[begin];
  
          for (ptrdiff_t i = begin; i < end; i++)
            used.reset((size_t)i);
        }
      };
    }
  }
  #endif
  #if defined(FERRET_MEMORY_POOL_SIZE) && !defined(FERRET_ALLOCATOR)
  
   #define FERRET_ALLOCATOR memory::allocator::pool
  
   #if !defined(FERRET_MEMORY_POOL_PAGE_TYPE)
    #define FERRET_MEMORY_POOL_PAGE_TYPE size_t
    #define FERRET_MEMORY_POOL_PAGE_COUNT                                   \
      (FERRET_MEMORY_POOL_SIZE / sizeof(FERRET_MEMORY_POOL_PAGE_TYPE))
   #else
    #define FERRET_MEMORY_POOL_PAGE_COUNT FERRET_MEMORY_POOL_SIZE
   #endif
  
  namespace memory{
    namespace allocator{
  
      memory_pool<FERRET_MEMORY_POOL_PAGE_TYPE, FERRET_MEMORY_POOL_PAGE_COUNT> program_memory;
  
      class pool{
      public:
  
        static void init(){ }
        
        template<typename FT>
        static inline void*  allocate(){ return program_memory.allocate(sizeof(FT)); }
        
        static inline void   free(void * ptr){ program_memory.free(ptr); }
      };
    }
  }
  #endif
  #ifdef FERRET_MEMORY_BOEHM_GC
  
  #define FERRET_ALLOCATOR memory::allocator::gc
  #define FERRET_DISABLE_RC true
  
  #include <gc.h>
  
  namespace memory{
    namespace allocator{
      
      class gc{
      public:
  
        static void init(){ GC_INIT(); }
        
        template<typename FT>
        static inline void* allocate(){
  #ifdef FERRET_DISABLE_MULTI_THREADING
          return GC_MALLOC(sizeof(FT));
  #else
          return GC_MALLOC_ATOMIC(sizeof(FT));
  #endif
        }
      
        static inline void  free(void * ptr){ }
      };
    }
  }
  #endif
  #if !defined(FERRET_ALLOCATOR)
  
  #define FERRET_ALLOCATOR memory::allocator::system
  
  namespace memory{
    namespace allocator{
  
      class system{
      public:
  
        static void init(){ }
  
        template<typename FT>
        static inline void* allocate(){ return ::malloc(sizeof(FT)); }
  
        static inline void  free(void * ptr){ ::free(ptr); } 
      };
    }
  }
  #endif
  namespace memory{
    namespace allocator{
      class synchronized{
        static mutex lock;
      public:
  
        static void init(){ FERRET_ALLOCATOR::init(); }
  
        template<typename FT>
        static inline void* allocate(){
          lock_guard guard(lock);
          return FERRET_ALLOCATOR::allocate<FT>();
        }
  
        static inline void  free(void * ptr){
          lock_guard guard(lock);
          FERRET_ALLOCATOR::free(ptr);
        }
      };
    }
  }
  #if  !defined(FERRET_DISABLE_MULTI_THREADING)
  
    #if defined(FERRET_MEMORY_POOL_SIZE) || defined(FERRET_HARDWARE_ARDUINO)
      mutex memory::allocator::synchronized::lock;
      #undef  FERRET_ALLOCATOR
      #define FERRET_ALLOCATOR memory::allocator::synchronized
    #endif
  
  #endif
  #if !defined(FERRET_RC_POLICY)
  namespace memory {
    namespace gc {
  
  #if !defined(FERRET_RC_TYPE)
    #define FERRET_RC_TYPE unsigned int
  #endif
      
  #if defined(FERRET_DISABLE_RC)
  
  #define FERRET_RC_POLICY memory::gc::no_rc
      
      class no_rc{
      public:
  
        inline void inc_ref() { }
        inline bool dec_ref() { return false; }
      };
  
  #else
  
      template<typename T>
      class rc{
      public:
        rc() : ref_count(0) {}
  
        inline void inc_ref() { ref_count++; }
        inline bool dec_ref() { return (--ref_count == 0); }
      
      private:
        T ref_count;
      };    
  
      #if defined(FERRET_DISABLE_MULTI_THREADING) || !defined(FERRET_STD_LIB)
        #define FERRET_RC_POLICY memory::gc::rc<FERRET_RC_TYPE>
      #endif
      
      #if defined(FERRET_STD_LIB) && !defined(FERRET_DISABLE_MULTI_THREADING)
        #define FERRET_RC_POLICY memory::gc::rc<::std::atomic<FERRET_RC_TYPE>>
      #endif
  #endif
    }
  }
  #endif
  template<typename>
  void type_id(){}
  
  using type_id_t = void(*)();
  typedef type_id_t type_t;
  
  class var;
  typedef var const & ref;
  class seekable_i;
  
  template <typename rc>
  class object_i : public rc{
  public:
    object_i() { }
    virtual ~object_i() { };
    
    virtual type_t type() const = 0;
    
  #if !defined(FERRET_DISABLE_STD_OUT)
    virtual void stream_console() const = 0;
  #endif
    
    virtual bool equals(ref) const;
  
    virtual seekable_i* cast_seekable_i() { return nullptr; }
  
    void* operator new(size_t, void* ptr){ return ptr; }
    void  operator delete(void * ptr){ FERRET_ALLOCATOR::free(ptr); }
  };
  
  typedef object_i<FERRET_RC_POLICY> object;
  #if !defined(FERRET_POINTER_T)
    #define FERRET_POINTER_T memory::pointer<object>
  #endif
  
  typedef FERRET_POINTER_T pointer_t;
  class var{
  public:
    explicit var(object* o = nullptr) : obj(o) { inc_ref(); }
  
    var(ref o)   : obj(o.obj) { inc_ref(); }
    // cppcheck-suppress useInitializationList
    var(var&& o) : obj(o.obj) { o.obj = nullptr; }
      
    ~var() { dec_ref(); }
  
    var& operator=(var&& other){
      if (this != &other){
        dec_ref();
        obj = other.obj;
        other.obj = nullptr;
      }
      return *this;
    }
    
    var& operator= (ref other){
      if (obj != other.obj){
        dec_ref();
        obj = other.obj;
        inc_ref();
      }
      return *this;
    }
  
    bool equals (ref) const;
  
    bool operator==(ref other) const { return equals(other); }
  
    bool operator!=(ref other) const { return !equals(other); }
    
    operator bool() const;
  
  #if !defined(FERRET_DISABLE_STD_OUT)
    void stream_console() const {
      if (obj != nullptr )
        obj->stream_console();
      else
        runtime::print("nil");
    }
  #endif
        
    inline object* get() const { return obj; }
    
    template<typename T>
    inline T* cast() const { return static_cast<T*>((object*)obj); }
  
    inline bool is_type(type_t type) const {
      return (static_cast<object*>(obj)->type() == type);
    }
  
    inline bool is_nil() const { return (obj == nullptr); }
  
  private:
    inline void inc_ref(){
  #if !defined(FERRET_DISABLE_RC)
      // Only change if non-null
      if (obj) obj->inc_ref();
  #endif
    }
      
    inline void dec_ref(){
  #if !defined(FERRET_DISABLE_RC)
      // Only change if non-null
      if (obj){
        // Subtract and test if this was the last pointer.
        if (obj->dec_ref()){
          delete obj;
          obj = nullptr;
        }
      }
  #endif
    }
      
    pointer_t obj;
  };
  
  template<>
  inline seekable_i* var::cast<seekable_i>() const { return obj->cast_seekable_i(); }
  
  template <typename rc>
  bool object_i<rc>::equals(ref o) const {
    return (this == o.get());
  }
  template<typename FT, typename... Args>
  inline var obj(Args... args) {
    void * storage = FERRET_ALLOCATOR::allocate<FT>();
    return var(new(storage) FT(args...));
  }
  
  inline var nil(){
    return var();
  }
}

// Runtime Prototypes
namespace ferret{
    namespace runtime {
      var list(ref v);
      var list(ref v);
      template <typename... Args>
      var list(ref first, Args const & ... args);
  
      var first(ref coll);
      var rest(ref coll);
      var cons(ref x, ref seq);
      var nth(ref seq, number_t index);
      var nthrest(ref seq, number_t index);
      size_t count(ref seq);
      bool is_seqable(ref seq);
    }
  
  #define for_each(x,xs) for(var tail = xs, x = runtime::first(xs);                 \
                             !tail.is_nil() && tail != cached::empty_sequence ;     \
                             tail = runtime::rest(tail), x = runtime::first(tail))
  template<typename T, typename... Args>
  inline var run(T const & fn, Args const & ... args);
        
  template<typename T>
  inline var run(T const & fn);
  
  template<>
  inline var run(ref);
}
