(ns indice.core
  (:refer-clojure :exclude [find])
  (:require [clojure.string :as str])
  (:import
    (org.apache.lucene.store Directory RAMDirectory FSDirectory)
    (clojure.lang Keyword)
    (org.apache.lucene.search TermQuery BooleanQuery$Builder Query BooleanClause$Occur SearcherManager SearcherFactory IndexSearcher ScoreDoc)
    (org.apache.lucene.index Term IndexWriter IndexWriterConfig IndexWriterConfig$OpenMode)
    (java.nio.file Paths)
    (org.apache.lucene.analysis.standard StandardAnalyzer)
    (org.apache.lucene.queryparser.classic QueryParser)
    (org.apache.lucene.document Document StoredField TextField StringField Field$Store)))

(defn- ^:static ^String stringify
  [s]
  (if (keyword? s)
    (.getName ^Keyword s)
    (str s)))

(def all (constantly true))

(defn new-snapshot
  "Create a new snapshot of `index` independant from the current one."
  [index]
  (assoc index :searcher (SearcherManager. ^Directory (:directory index) (SearcherFactory.))))

(defn index
  "Open or create an index on the specificed Directory"
  [^Directory directory & [options]]
  (let [analyzer (or (:analyzer options) (StandardAnalyzer.))
        defaultField (or (:defaultField options) "text")
        config (-> (IndexWriterConfig. analyzer)
                   (.setOpenMode (if (:forceCreate options)
                                   IndexWriterConfig$OpenMode/CREATE
                                   IndexWriterConfig$OpenMode/CREATE_OR_APPEND)))
        writer (IndexWriter. directory config)]
    (.commit writer)
    (->
      {:directory directory
       :analyzer  analyzer
       :parser    (QueryParser. defaultField analyzer)
       :writer    writer
       :indexed   (or (:indexed options) all)
       :stored    (or (:stored options) all)
       :tokenized (or (:tokenized options) all)
       :max-count (or (:max-count options) 100)}
      new-snapshot)))

(defn ram-index
  "Create a new in-memory index."
  [& [options]]
  (index (RAMDirectory.) options))

(defn fs-index
  "Open or create an index on a filesystem"
  [^String path & [options]]
  (index (FSDirectory/open (Paths/get path)) options))

(defn update-snapshot
  "Undate the current snapshot for searching the index"
  [index]
  (.maybeRefreshBlocking (:searcher index)))

(defn close-snapshot
  "Release the snaoshot of `index`"
  [index]
  (.close (:searcher index)))

(defn write
  "Write to `index` the document specificed by a field->value map `m`"
  [index m]
  (let [doc (Document.)]
    (doseq [[k ^String v] m]
      (let [s (stringify k)
            v (str v)
            field (if ((:indexed index) k)
                    (let [stored ^Field$Store (if ((:stored index) k)
                                                Field$Store/YES
                                                Field$Store/NO)]
                      (if ((:tokenized index) k)
                        (TextField. s v stored)
                        (StringField. s v stored)))
                    (StoredField. s v))]
        (.add doc field)))
    (.addDocument ^IndexWriter (:writer index) doc)
    m))

(defn commit-index
  "Force the immediate commit of the documents added to `index`"
  [index]
  (.commit ^IndexWriter (:writer index)))

; -- queries

(defn ^Query parse-query
  "Build a query from a string using the standard Lucene query syntax"
  [index ^String q]
  (.parse ^QueryParser (:parser index) q))

(defn ^TermQuery term
  "Build a term query that accepts docs where field = value"
  [field ^String value]
  (TermQuery. (Term. (stringify field) value)))

(declare all-of)

(defn ^Query querify
  "Turn a field->value map into a all-of query"
  [q]
  (cond
    (instance? Query q) q
    (and (map? q) (not (empty? q))) (let [clauses (map (fn [[k v]] (term k v)) q)]
                                      (if (= 1 (count clauses))
                                        (first clauses)
                                        (apply all-of clauses)))
    :else (throw (IllegalArgumentException. "Should be a non-empty map or a Query."))))

(defn- boolean-query
  [^long min ^BooleanClause$Occur mode qs]
  (let [b (BooleanQuery$Builder.)]
    (.setMinimumNumberShouldMatch b min)
    (doseq [q qs] (.add b (querify q) mode))
    (.build b)))

(defn all-of
  "Accept docs that satisfy all the sub-queries"
  [& clauses]
  (boolean-query 0 BooleanClause$Occur/MUST clauses))

(defn one-of
  "Accept docs that satisfy at least one of the sub-queries"
  [& clauses]
  (boolean-query 1 BooleanClause$Occur/SHOULD clauses))

(defn none-of
  "Accept docs that satisfy none of the sub-queries"
  [& clauses]
  (boolean-query 0 BooleanClause$Occur/MUST_NOT clauses))

(defn- find*
  [index select-fn query & [max-count]]
  (let [^int max-count (or max-count (:max-count index))
        ^SearcherManager manager (:searcher index)
        ^IndexSearcher searcher (.acquire manager)
        ^Query query (querify query)]
    (try
      (let [results (.search searcher query max-count)
            hit-count (.totalHits results)
            hits (.scoreDocs results)]
        (println query)
        (with-meta
          (doall (map (fn [^ScoreDoc sDoc] (select-fn (.doc searcher (.doc sDoc)))) hits))
          {:count hit-count}))
      (finally (.release manager searcher)))))

(defn- get-field
  [doc n]
  (when-let [f (.getField doc n)]
    (let [v (.stringValue f)]
      (if (str/starts-with? v ":")
        (keyword (subs v 1))
        v))))

(defn find
  "Return a list of the first `max-count` documents found by `query`, as maps,
  retireving only the fields named in `select`."
  [index select query & [max-count]]
  (let [select1 (map stringify select)
        select-fn
        (fn [doc]
          (when doc
            (into
              {}
              (map (fn [k s] [k (get-field doc s)]) select select1))))]
    (find* index select-fn query max-count)))

(defn find-tuples
  "Return a list of the first `max-count` documents found by `query`,
  as vectors of field values, retireving only the fields named in `select`."
  [index select query & [max-count]]
  (let [select (map stringify select)
        select-fn
        (fn [doc]
          (when doc
            (into
              []
              (map (fn [s] (get-field doc s))) select)))]
    (find* index select-fn query max-count)))

(defn find-values
  "Return a list of the values of the field `select` of the first `max-count`
  documents found by `query`."
  [index select query & [max-count]]
  (let [select (stringify select)
        select-fn
        (fn [doc]
          (when doc
            (get-field doc select)))]
    (find* index select-fn query max-count)))

(defn delete
  "Delete from index documents that are matched by one of the queries"
  [index & queries]
  (.deleteDocuments ^IndexWriter (:writer index) #^"[Lorg.apache.lucene.search.Query;" (into-array Query queries)))

(defn close-index
  [index]
  (close-snapshot index)
  (.close (:writer index))
  (.close (:directory index)))
