(ns struktur.jwt
  (:require
   [buddy.core.keys :as keys]
   [buddy.sign.jwt :as jwt]
   [clojure.spec.alpha :as s]
   [com.stuartsierra.component :as c]))

(s/def ::sha-algs #{:hs256 :hs512})
(s/def ::asymetric-algs #{:es256 :es512 :ps256 :ps512 :rs256 :rs512})
(s/def ::algs (s/or :sha ::sha-algs :asymetric ::asymetric-algs))

(defprotocol JwtEncoder
  (encode [this data])
  (decode [this data]))

(defrecord SHASigner [config]
  JwtEncoder
  (encode [this data]
    (jwt/sign data (:secret config) (dissoc config :secret)))
  (decode [this data]
    (jwt/unsign data (:secret config) (dissoc config :secret))))

(defn new-sha-signer
  [config]
  (map->SHASigner {:config config}))

(defrecord AsymetricSigner [config public-key private-key]
  c/Lifecycle
  (start [this]
    (let [public-key  (keys/public-key (-> config :keypair :public))
          private-key (keys/private-key (-> config :keypair :private))]
      (assoc this :public-key public-key :private-key private-key)))
  (stop [this]
    (assoc this :public-key nil :private-key nil))

  JwtEncoder
  (encode [this data]
    (jwt/sign data private-key (dissoc config :keypair)))
  (decode [this data]
    (jwt/unsign data public-key (dissoc config :keypair))))

(defn new-asymetric-signer
  [config]
  (map->AsymetricSigner {:config config}))

(defn new-jwt-encoder
  [{:keys [alg] :as config}]
  (let [result (s/conform ::algs alg)]
    (if (s/invalid? result)
      (throw (ex-info "unsupported algorithm" {:alg alg}))
      (case (first result)
        :sha (new-sha-signer config)
        :asymetric (new-asymetric-signer config)))))
