(ns struktur.jwt
  (:require
   [buddy.core.keys :as keys]
   [buddy.sign.jwt :as jwt]
   [clojure.spec.alpha :as s]
   [com.stuartsierra.component :as c]
   [defn-spec.core :refer [defn-spec]]))

(s/def :sha-signer/algorithm #{:hs256 :hs512})
(s/def :asymetric-signer/algorithm
  #{:es256 :es512 :ps256 :ps512 :rs256 :rs512})
(s/def ::algorithm
  (s/or :sha :sha-signer/algorithm
        :asymetric :asymetric-signer/algorithm))

(s/def ::secret some?)
(s/def ::sha-signer-config
  (s/keys :req-un [:sha-signer/algorithm]
          :opt-un [::secret]))

(s/def ::public string?)
(s/def ::private string?)
(s/def ::keypair (s/keys :req-un [::public ::private]))
(s/def ::asymetric-signer-config
  (s/keys :req-un [:asymetric-signer/algorithm ::keypair]))

(defprotocol JwtEncoder
  (encode [this data])
  (decode [this data]))

(defrecord SHASigner [config]
  JwtEncoder
  (encode [this data]
    (jwt/sign data (:secret config) {:alg (:algorithm config)}))
  (decode [this data]
    (jwt/unsign data (:secret config) {:alg (:algorithm config)})))

(defn-spec new-sha-signer
  {::s/args (s/cat :config ::sha-signer-config)}
  [config]
  (map->SHASigner {:config config}))

(defrecord AsymetricSigner [config public-key private-key]
  c/Lifecycle
  (start [this]
    (let [public-key  (keys/public-key (-> config :keypair :public))
          private-key (keys/private-key (-> config :keypair :private))]
      (assoc this :public-key public-key :private-key private-key)))
  (stop [this]
    (assoc this :public-key nil :private-key nil))

  JwtEncoder
  (encode [this data]
    (jwt/sign data private-key {:alg (:algorithm config)}))
  (decode [this data]
    (jwt/unsign data public-key {:alg (:algorithm config)})))

(defn-spec new-asymetric-signer
  {::s/args (s/cat :config ::asymetric-signer-config)}
  [config]
  (map->AsymetricSigner {:config config}))

(defn new-jwt-encoder
  [{:keys [algorithm] :as config}]
  (let [result (s/conform ::algorithm algorithm)]
    (when-not (s/invalid? result)
      (case (first result)
        :sha (new-sha-signer config)
        :asymetric (new-asymetric-signer config)))))
