(ns clj.faris.qed.resource.normal
  (:require [clj.faris.qed.system.mongo.db-io :as io]
            [monger.collection :as mc]
            [clj.faris.qed.resource.util :as util])
  (:use clj.faris.qed.resource.components
        [liberator.core :only [defresource]]))

(defresource main-resource
  [file-name]
  html-types
  :allowed-methods [:get]
  :handle-ok (fn [_] (slurp file-name)))

(defn get-list-resource
  [db-instance mapper coll-name]
  (merge json-types
         {:handle-ok (fn [context]
                       (let [params (-> context :request :params)
                             opts (select-keys params [:page :per-page :order])
                             kwargs (dissoc params :page :per-page :order)
                             entities (io/find-all db-instance coll-name
                                                   kwargs
                                                   opts)
                             result (map mapper entities)]
                         result))}))

(defn- get-one-resource
  [db-instance mapper exists-fn coll-name identifier]
  (merge json-types
         (exists-fn db-instance coll-name identifier)
         {:handle-ok #(-> % :entity mapper)}))

(defn get-one-by-id-resource
  [db-instance mapper coll-name id]
  (get-one-resource db-instance mapper exists-by-id? coll-name id))

(defn get-one-by-slug-resource
  [db-instance mapper coll-name slug]
  (get-one-resource db-instance mapper exists-by-slug? coll-name slug))

(defn- put-one-resource
  [db-instance mapper validator exists-fn update-fn coll-name identifier]
  (merge json-types
         (body-malformed? mapper validator)
         (exists-fn db-instance coll-name identifier)
         {:can-put-to-missing? false
          :put! (fn [context]
                  (let [body (:body context)]
                    (update-fn db-instance
                               coll-name
                               identifier
                               body)))}))

(defn put-one-by-id-resource
  [db-instance mapper validator coll-name id]
  (put-one-resource db-instance
                    mapper
                    validator
                    exists-by-id?
                    mc/update-by-id
                    coll-name
                    id))

(defn put-one-by-slug-resource
  [db-instance mapper validator coll-name slug]
  (put-one-resource db-instance
                    mapper
                    validator
                    exists-by-slug?
                    io/update-by-slug
                    coll-name
                    slug))

(defn- delete-one-resource
  [db-instance exists-fn entity-fn update-fn remove-fn coll-name identifier]
  (merge json-types
         (exists-fn db-instance coll-name identifier)
         {:delete! (fn [context]
                     (let [entity (entity-fn db-instance
                                             coll-name
                                             identifier)
                           deleted (:deleted entity)]
                       (if (false? deleted)
                         (update-fn db-instance
                                    coll-name
                                    identifier
                                    (assoc entity :deleted true))
                         (remove-fn db-instance
                                    coll-name
                                    identifier))))}))

(defn delete-one-by-slug-resource
  [db-instance coll-name slug]
  (delete-one-resource db-instance
                       exists-by-slug?
                       util/get-entity-by-slug
                       io/update-by-slug
                       io/remove-by-slug
                       coll-name
                       slug))

(defn delete-one-by-id-resource
  [db-instance coll-name id]
  (delete-one-resource db-instance
                       exists-by-id?
                       util/get-entity-by-id
                       mc/update-by-id
                       mc/remove-by-id
                       coll-name
                       id))

(defn post-one-resource
  [db-instance jsonable-mapper saveable-mapper validator coll-name]
  (merge json-types
         (body-malformed? saveable-mapper validator)
         {:post! (fn [context]
                   (let [body (-> context :body saveable-mapper)
                         entity (mc/insert-and-return db-instance
                                                      coll-name
                                                      body)
                         result {:entity entity}]
                     result))
          :handle-created #(-> % :entity jsonable-mapper)}))
