(ns clj.faris.qed.resource.components
  (:require [cemerick.friend :as friend]
            [clj.faris.qed.resource.util :as util]))

(def json-types
  {:available-media-types ["application/json"]})

(def html-types
  {:available-media-types ["text/html"]})

(defn base-auth
  [roles]
  {:authorized? (fn [ctx]
                  (let [identity (-> ctx :request friend/identity)
                        result (if (nil? identity)
                                 [false {}]
                                 [true {:identity identity}])]
                    result))
   :handle-unauthorized (fn [ctx]
                          (friend/throw-unauthorized nil
                                                     {:wrapped-handler base-auth}))
   :allowed? (fn [ctx] (->> ctx
                           :request
                           (friend/authorized? roles)))
   :handle-forbidden (fn [ctx]
                       (friend/throw-unauthorized (:identity ctx)
                                                  {:wrapped-handler base-auth}))})

(def admin-auth
  (base-auth #{:admin}))

(defn- owner-auth
  [db-instance roles entity-fn coll-name identifier]
  (update-in base-auth
             [:allowed?]
             (fn [f]
               (fn [ctx]
                 (let [role-result (f ctx)
                       result (if (nil? role-result)
                                [false {}]
                                (let [entity (entity-fn ctx
                                                        db-instance
                                                        coll-name
                                                        identifier)
                                      creator-slug (-> entity :creator :slug)
                                      user-slug (-> ctx :request :user :slug)
                                      allowed? (and (not (or (nil? user-slug)
                                                             (nil? creator-slug)))
                                                    (= user-slug creator-slug))]
                                  [allowed? {:entity entity}]))]
                   result)))))

(defn owner-auth-by-slug
  [db-instance roles coll-name slug]
  (owner-auth db-instance roles util/get-entity-by-slug coll-name slug))

(defn owner-auth-by-id
  [db-instance roles coll-name id]
  (owner-auth db-instance roles util/get-entity-by-id coll-name id))

(defn body-malformed?
  [mapper validator]
  {:malformed? (fn [ctx]
                 (let [body (-> ctx :request :body mapper)
                       error (validator body)
                       result (if (nil? error)
                                [false {:body body}]
                                [true {:error error}])]
                   result))
   :handle-malformed :error})

(defn- exists?
  [db-instance entity-fn coll-name identifier]
  {:exists? (fn [ctx]
              (let [entity (entity-fn ctx db-instance coll-name identifier)
                    result (if (nil? entity)
                             [false {}]
                             [true {:entity entity}])]
                result))})

(defn exists-by-slug?
  [db-instance coll-name slug]
  (exists? db-instance util/get-entity-by-slug coll-name slug))

(defn exists-by-id?
  [db-instance coll-name id]
  (exists? db-instance util/get-entity-by-id coll-name id))
