(ns clj.faris.qed.resource
  (:require [cemerick.friend :as friend]
            [monger.collection :as mc]
            [clj.faris.qed.system.mongo.db-io :as io])
  (:use [liberator.core :only [defresource]]))

(defresource main-resource
  [file-name]
  {:available-media-types ["text/html"]
   :allowed-methods [:get]
   :handle-ok (fn [_]
                (slurp file-name))})

(def -media-types
  {:available-media-types ["application/json"]})

(defn base-auth
  [roles]
  {:authorized? (fn [context]
                  (let [identity (friend/identity (:request context))
                        result (when-not (nil? identity)
                                 {:identity identity})]
                    result))
   :handle-unauthorized (fn [context]
                          (friend/throw-unauthorized nil
                                                     {:wrapped-handler base-auth}))
   :allowed? (fn [context]
               (friend/authorized? roles (:request context)))
   :handle-forbidden (fn [context]
                       (friend/throw-unauthorized (:identity context)
                                                  {:wrapped-handler base-auth}))})

(defn owner-auth
  [db-instance roles coll-name slug]
  (update-in base-auth
             [:allowed?]
             (fn [f]
               (fn [context]
                 (let [role-result (f context)
                       result (when-not (nil? role-result)
                                (let [entity (:entity
                                              context
                                              (io/find-one-by-slug db-instance
                                                                   coll-name
                                                                   slug))
                                      creator-slug (-> entity :creator :slug)
                                      user-slug (-> context :request :user :slug)]
                                  [(and (not (or (nil? user-slug) (nil? creator-slug)))
                                        (= user-slug creator-slug))
                                   {:entity entity}]))]
                   result)))))

(defn body-malformed?
  [mapper validator]
  {:malformed? (fn [context]
                 (let [body (-> context :request :body mapper)
                       error (validator body)
                       result (if (nil? error)
                                [false {:body body}]
                                [true {:error error}])]
                   result))
   :handle-malformed :error})

(defn exists?
  [db-instance coll-name slug]
  {:exists? (fn [context]
              (let [entity (:entity
                            context
                            (io/find-one-by-slug db-instance
                                                 coll-name
                                                 slug))
                    result (when-not (nil? entity)
                             {:entity entity})]
                result))})

(defn get-one-by-slug-resource
  [mapper coll-name slug]
  (merge (exists? coll-name slug)
         {:handle-ok #(-> % :entity mapper)}))

(defn get-list-resource
  [db-instance mapper coll-name]
  {:handle-ok (fn [context]
                (let [params (-> context :request :params)
                      opts (select-keys params [:page :per-page :order])
                      kwargs (dissoc params :page :per-page :order)
                      entities (io/find-all db-instance coll-name
                                            kwargs
                                            opts)
                      result (map mapper entities)]
                  result))})

(defn post-one-resource
  [db-instance jsonable-mapper saveable-mapper validator coll-name]
  (merge (body-malformed? saveable-mapper validator)
         {:post! (fn [context]
                   (let [body (-> context :body saveable-mapper)
                         entity (mc/insert-and-return db-instance
                                                      coll-name
                                                      body)
                         result {:entity entity}]
                     result))
          :handle-created #(-> % :entity jsonable-mapper)}))

(defn put-one-by-slug-resource
  [db-instance mapper validator coll-name slug]
  (merge (body-malformed? mapper validator)
         (exists? coll-name slug)
         {:can-put-to-missing? false
          :put! (fn [context]
                  (let [body (:body context)]
                    (io/update-by-slug db-instance
                                       coll-name
                                       slug
                                       body)))}))

(defn delete-one-by-slug-resource
  [db-instance coll-name slug]
  (merge (exists? coll-name slug)
         {:delete! (fn [context]
                     (let [entity (:entity
                                   context
                                   (io/find-one-by-slug db-instance
                                                        coll-name
                                                        slug))
                           deleted (:deleted entity)]
                       (if (false? deleted)
                         (io/update-by-slug db-instance
                                            coll-name
                                            slug
                                            (assoc entity :deleted true))
                         (io/remove-by-slug db-instance
                                            coll-name
                                            slug))))}))
