(ns clj.faris.qed.auth.middleware
  (:require [clj.faris.qed.auth.core :as ac]
            [clj.faris.qed.util :as util]))

(defn -add-user-cookie-middleware!
  [handler]
  (fn [request]
    (let [user-cookie (-> request :user-state :cookies)
          _ (ac/add-user-cookie! user-cookie)]
      (handler request))))

(defn -user-transform-middleware
  [handler]
  (fn [request]
    (let [users (:users request)
          user-collection (ac/->user-collection users)
          user-cookie (ac/->user-cookie users)
          new-request (if (or (nil? user-collection)
                              (nil? user-cookie))
                        request
                        (-> request
                            (assoc :user-state {:collection user-collection
                                                :cookies user-cookie})
                            (dissoc :users)))]
      (handler new-request))))

(defn -user->cookie-middleware
  [handler]
  (fn [request]
    (let [user (:user request)
          response (handler request)
          new-response (if (and (not (ac/logged-in? request))
                                (map? user))
                         (-> response
                             (assoc-in [:cookies :__ID__]
                                       (ac/->user-cookie-key user))
                             (dissoc :user))
                         response)]
      new-response)))

(defn -user-cookie->auth-header-middleware
  [handler]
  (fn [request]
    (let [cookie (-> request :cookies :__ID__)
          header-value (when (and (ac/logged-in? request)
                                  (string? cookie)
                                  (not (empty? cookie)))
                         (->> cookie
                              ac/user-cookie->user-data
                              ac/user-data->auth-header-value))
          new-request (if (not (empty? header-value))
                        (assoc-in request [:headers "authorization"] header-value)
                        request)]
      (handler new-request))))

(defn -cookie-as-bank-middleware
  [handler]
  (fn [request]
    (let [users (:users request)
          new-users (if (and (ac/logged-in? request)
                             (nil? users))
                      (let [result (ac/user-cookie->user-data (-> request
                                                                  :cookies
                                                                  :__ID__))]
                        (when-not (nil? result)
                          [result]))
                      users)
          new-request (if (nil? new-users)
                        request
                        (assoc request :users new-users))]
      (handler new-request))))

(defn -map-only-middleware
  [handler]
  (fn [request]
    (if (map? request)
      (handler request)
      (handler {}))))

(defn email->username-middleware
  [handler]
  (fn [request]
    (let [email (-> request :params :email)
          new-request (if-not (nil? email)
                        (-> request
                            (assoc-in [:params :username] email)
                            (util/dissoc-in [:params :email]))
                        request)]
      (handler new-request))))

;; dipake abis friend-middleware
(defn extract-user-info-middleware
  [handler]
  (fn [request]
    (-> request
        ac/extract-user-info
        handler)))

(defn user-middleware
  [handler]
  (-> handler
      -add-user-cookie-middleware!
      -user-transform-middleware
      -cookie-as-bank-middleware
      -user-cookie->auth-header-middleware
      email->username-middleware
      -user->cookie-middleware
      -map-only-middleware))
