(ns clj.faris.qed.validator.rule
  (:require [clojure [string :as string]]))

(defmacro defrule
  [name & args]
  (let [default-settings {:force? false
                          :message (fn [value] (str "Value is %s" value))}
        [docstring args] (if (string? (first args))
                           [(first args) (next args)]
                           [nil args])
        [settings args] (if (map? (first args))
                          [(into default-settings (first args)) (next args)]
                          [default-settings args])
        [args body] [(first args) (next args)]]
    `(do
       (def ~name
         (with-meta
           (fn [~@args] ~@body)
           ~settings))
       (alter-meta! (var ~name)
                    assoc
                    :doc ~docstring
                    :arglists '([~@args])))))

(defn process-rule
  [rule-and-args]
  (let [[rule args] [(first rule-and-args) (next rule-and-args)]
        [args message] (split-with #(not= :message %) args)
        message (second message)
        rule-settings (if (nil? message)
                        (meta rule)
                        (assoc (meta rule) :message message))
        meta-ed-rule (with-meta rule rule-settings)]
    meta-ed-rule))

(defrule required
  "Value must not be nil."
  {:force? true
   :message (fn [value]
              "Value cannot be nil.")}
  [value]
  (not (nil? value)))

(defrule is-positive
  "The given value should have a positive numeric value."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a number and have a positive value or zero."))}
  [value]
  (and (number? value) (or (= value 0) (> value 0))))

(defrule is-negative
  "The given value should have a positive numeric value."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a number and have a negative value."))}
  [value]
  (and (number? value) (< value 0)))

(defrule min-value-threshold
  "Sets the acceptable minimal value of a numeric."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be above or equal threshold."))}
  [threshold value]
  (and (number? value) (or (= value threshold) (> value threshold))))

(defrule max-value-threshold
  "Sets the acceptable maximal value of a numeric."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be below or equal threshold."))}
  [threshold value]
  (and (number? value) (or (= value threshold) (< value threshold))))

(defrule is-coll
  "Make sure the value is a collection."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a collection, but its type is " (type value) "."))}
  [value]
  (coll? value))

(defrule is-vec
  "Make sure the value is a vector."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a vector, but its type is " (type value) "."))}
  [value]
  (vector? value))

(defrule is-map
  "Make sure the value is a map."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a map, but its type is " (type value) "."))}
  [value]
  (map? value))

(defrule is-not-empty
  "Check if the given value is a collection or a string, and make sure it is not empty."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a collection and have at least 1 item."))}
  [value]
  (and (or (string? value) (coll? value)) (boolean (seq value))))

(defrule any
  "Check if the given value is a collection, not empty, and one
   of its collection gives true when used as an input for the given
   function."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a collection and one of its items gives true with the given function."))}
  [func value]
  (and (coll? value) (not (not-any? func value))))

(defrule every
  "Check if the given value is a collection, not empty, and every
   of its collection give true when used as ana input for the given
   function."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "It should be a collection and every of its items gives true with the given function."))}
  [func value]
  (and (coll? value) (every? func value)))

(defrule is-str
  "Check if the given values has type String."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "Its type should be a string, while its type is " (type value) "."))}
  [value]
  (string? value))

(defrule is-int
  "Check if the given value has type Integer."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "Its type should be a string, while its type is " (type value) "."))}
  [value]
  (integer? value))

(defrule regex
  "Check if the given value match with the given regex."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "Its type should be a string and match with the given regex."))}
  [regex-config value]
  (and (string? value) (boolean (re-matches regex-config value))))

(defrule is-email
  "Check if the given value is an email."
  {:message (fn [value]
              (str "Value is "
                   value ". "
                   "Its type should be a string and match with the form of an email."))}
  [value]
  (regex #"[a-zA-Z0-9.+_-]+@[a-zA-Z0-9.+_-]+\.[a-zA-Z]{2,4}" value))
