(ns farbetter.utils)

;;;;;;;;;;; Exception handling

(defmacro throw-far-error [description error-type error-subtype error-map]
  `(let [emap# (merge ~error-map
                      {:type ~error-type :subtype ~error-subtype})]
     (throw (ex-info (str ~description ". error-map: " emap#)
                     emap#))))

;; This macro asserts that the body throws a particular far-error
;; Must be used inside a deftest block
(defmacro throws [type subtype & body]
  `(try
     ~@body
     (throw-far-error "Did not throw" :did-not-throw :did-not-throw {})
     (catch clojure.lang.ExceptionInfo e#
       (let [data# (ex-data e#)]
         (clojure.test/is (= ~type (:type data#)))
         (clojure.test/is (= ~subtype (:subtype data#)))))))


;;;;;;;;;;; Environment variables

(defn construct-java-type-from-str [klass str]
  (.newInstance
   (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
   (object-array [str])))

(defn construct-from-str [var-type var-value]
  (if (= clojure.lang.Keyword var-type)
    (keyword var-value)
    (construct-java-type-from-str var-type var-value)))

(defn get-env-var
  ([var-name var-type default-value]
   (if-let [var-value-str (System/getenv var-name)]
     (construct-from-str var-type var-value-str)
     default-value))

  ([var-name var-type]
   (if-let  [var-value (get-env-var var-name var-type nil)]
     var-value
     (throw-far-error "Environment variable not found"
                      :environment-error :env-var-not-found
                      {:var-name var-name :var-type var-type}))))
