(ns farbetter.utils
  (:refer-clojure :exclude [long])
  (:require
   #?(:cljs cljsjs.long)
   #?(:cljs [cljs.pprint :refer [pprint]])
   #?(:cljs [cljs-time.extend]) ;; to make = work as expected
   [#?(:clj clj-time.core :cljs cljs-time.core) :as t]
   [#?(:clj clj-time.format :cljs cljs-time.format) :as f]
   #?(:clj [clj-uuid :as clj-uuid])
   [#?(:clj clojure.core.async :cljs cljs.core.async)
    :refer [alts! take! timeout #?@(:clj [go <!!])]]
   [clojure.string :as string :refer [join]]
   #?(:clj [clojure.test :refer [is]] :cljs [cljs.test :as test])
   [cognitect.transit :as transit]
   #?(:cljs [murmur])
   #?(:clj [puget.printer :refer [cprint]])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debug errorf infof]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [cljs.test :refer [async is]]
      [farbetter.utils :refer [inspect sym-map throws]]))
  #?(:clj
     (:import [com.google.common.hash Hashing]
              [com.google.common.primitives UnsignedLong]
              [java.io ByteArrayInputStream ByteArrayOutputStream]
              [java.nio.charset StandardCharsets]
              [java.util UUID]
              [org.joda.time DateTime])
     :cljs
     (:import [goog.date Date UtcDateTime])))


#?(:cljs (def class type))

;; TODO: Add schemas to these functions

;;;;;;;;;;;;;;;; Handy utilities

;; dissoc-in function was stolen from:
;; https://github.com/clojure/core.incubator/blob/master/src
;; /main/clojure/clojure/core/incubator.clj
(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

;;;;;;;;;;; Exception handling and testing

(defn throw-far-error
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [desc error-type error-subtype error-map]
  (let [error-map (merge error-map
                         {:type error-type :subtype error-subtype})
        desc (cond-> desc
               (not= "." (str (last desc))) (str "."))]
    (throw (ex-info (str desc " error-map: " error-map)
                    error-map))))

(defn abs
  "Returns the absolute value of n"
  [n]
  (if (neg? n)
    (- n)
    n))

(defn within?
  "Tests if the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin expected actual]
  (let [neg-inf #?(:clj
                   (Double/NEGATIVE_INFINITY)
                   :cljs
                   (.-NEGATIVE_INFINITY js/Number))
        pos-inf #?(:clj
                   (Double/POSITIVE_INFINITY)
                   :cljs
                   (.-POSITIVE_INFINITY js/Number))]
    (cond
      (= expected neg-inf) (= actual neg-inf)
      (= expected pos-inf) (= actual pos-inf)
      :else (-> actual
                (- expected)
                (abs)
                (<= error-margin)))))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (defn construct-java-type-from-str [klass s]
     (.newInstance
      (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))
#?(:clj
   (defn construct-from-str [var-type var-value]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (defn get-env-var
     ([var-name var-type default-value]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))

     ([var-name var-type]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))))

;;;;;;;;;;; Time functions

(defn get-current-time-ms []
  #?(:clj (System/currentTimeMillis)
     :cljs (.getTime (js/Date.))))

(defn str->dt
  " Converts an ISO8601-formatted string to a date-time object"
  [s]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/parse formatter s)))

(defn dt->str
  "Converts a date-time object to an ISO8601-formatted string"
  [dt]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/unparse formatter dt)))

(defn add-days
  "Adds the specifed number of days to an ISO8601-formatted string
   Returns an ISO8601-formatted string"
  [dt-str days]
  (-> dt-str
      str->dt
      (t/plus (t/days days))
      dt->str))

(defn day-at
  "Takes a date-time, an hour, and a minute
   Returns a new date-tiem with the same date, but the
   specified hour and minute."
  ([dt] (day-at dt 0 0))
  ([dt h] (day-at dt h 0))
  ([dt h m]
   (t/date-time (t/year dt) (t/month dt) (t/day dt) h m)))

(defn get-num-midnights
  "Return the number of midnights (actually 23:59:59s) between two date-times"
  [begin end]
  (loop [night (t/date-time (t/year begin) (t/month begin) (t/day begin)
                            23 59 59)
         count 0]
    (if (t/within? begin end night)
      (recur (t/plus night (t/days 1)) (inc count))
      count)))

(defn dt+days
  "Takes a date-time, a number of days, and optionally hour and minute.
   Returns a new date-time offset by the number of days, with hour and
   minute set if they were provided."
  ([dt n]
   (dt+days dt n 0 0))
  ([dt n h]
   (dt+days dt n h 0))
  ([dt n h m]
   (day-at (t/plus dt (t/days n)) h m)))

(defn hrs-diff [dt1 dt2]
  (/ (t/in-seconds (t/interval dt1 dt2)) 3600.0))

(defn local->utc
  "Takes a local date-time, hours, and (optionally) minutes offset from UTC.
  Returns the corresponding date-time in UTC."
  ([dt offset-hrs]
   (local->utc dt offset-hrs 0))
  ([dt offset-hrs offset-mins]
   (t/minus dt (t/hours offset-hrs) (t/minutes offset-mins))))

(defn same-day? [dt1 dt2]
  (= [(t/year dt1) (t/month dt1) (t/day dt1)]
     [(t/year dt2) (t/month dt2) (t/day dt2)]))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;;;;;;;;;;;;;;; Randomness

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))

;;;;;;;;; Serialization / Deserialization ;;;;;;;;;

(def initial-transit-buffer-size 4096)
(def date-time-formatter (f/formatters :date-hour-minute-second-ms))
(def date-time-transit-tag "dt")

(def date-time-writer
  (transit/write-handler
   (constantly date-time-transit-tag)
   #(f/unparse date-time-formatter %)))

(def date-time-reader
  (transit/read-handler
   #(f/parse date-time-formatter %)))

(defn edn->transit [edn]
  #?(:clj
     (let [out (ByteArrayOutputStream. initial-transit-buffer-size)
           writer (transit/writer
                   out :json
                   {:handlers {DateTime date-time-writer}})]
       (transit/write writer edn)
       (.toString ^ByteArrayOutputStream out "UTF-8"))
     :cljs
     (transit/write (transit/writer
                     :json
                     {:handlers {UtcDateTime date-time-writer}})
                    edn)))

(defn transit->edn [^String transit-str]
  (when transit-str
    #?(:clj
       (let [bytes (.getBytes transit-str "UTF-8")
             in (ByteArrayInputStream. bytes)
             reader (transit/reader
                     in :json
                     {:handlers {date-time-transit-tag date-time-reader}})]
         (transit/read reader))
       :cljs
       (transit/read (transit/reader
                      :json
                      {:handlers {date-time-transit-tag date-time-reader}})
                     transit-str))))

;;;;;;;;;;;;;;;;;;;; Hashing ;;;;;;;;;;;;;;;;;;;;

(s/defn murmur-hash :- s/Int
  [s :- s/Str]
  #?(:clj
     (-> (Hashing/murmur3_32)
         (.hashBytes (.getBytes ^String s) 0 (count s))
         (.asInt))
     :cljs
     (murmur/hashBytes s (count s) 0)))

;;;;;;;;;;;;;;;;;;;; byte-array for cljs ;;;;;;;;;;;;;;;;;;;;

#?(:cljs
   (defn byte-array
     ([size-or-seq]
      (if (sequential? size-or-seq)
        (byte-array (count size-or-seq) size-or-seq)
        (byte-array size-or-seq 0)))
     ([size init-val-or-seq]
      (let [ba (js/Int8Array. size)]
        (if (sequential? init-val-or-seq)
          (.set ba (clj->js init-val-or-seq))
          (.fill ba init-val-or-seq))
        ba))))

(def byte-array-type
  #?(:clj
     (class (byte-array []))
     :cljs
     js/Int8Array))

(defn byte-array? [x]
  (when-not (nil? x)
    (= byte-array-type (class x))))

;;;;;;;;;;;;;;;;;;;; Longs ;;;;;;;;;;;;;;;;;;;;

#?(:cljs (def Long js/Long))

(s/defn long? :- s/Bool
  [x :- s/Any]
  (when-not (nil? x)
    (= Long (class x))))

(s/defn long :- Long
  [x :- s/Any]
  (when-not (nil? x)
    #?(:clj (Long. x) :cljs (Long.fromValue x))))

(s/defn long= :- s/Bool
  [a :- s/Any
   b :- s/Any]
  #?(:clj
     (= a b)
     :cljs
     (.equals a b)))

(s/defn hex-str->long :- Long
  [hex-str :- s/Str]
  #?(:clj
     (.longValue (UnsignedLong/valueOf hex-str 16))
     :cljs
     (.fromString Long hex-str 16)))

(s/defn long->hex-str :- s/Str
  [l :- Long]
  #?(:clj
     (Long/toHexString l)
     :cljs
     (-> (.toUnsigned l)
         (.toString 16))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn inspect-helper [& exprs]
  (doseq [[expr-name expr-val] exprs]
    (println (str expr-name ":"))
    (#?(:clj cprint :cljs pprint) expr-val)))

(defn throws-helper
  [error-type error-subtype body]
  (try
    (body)
    (throw-far-error "Did not throw" :execution-error :did-not-throw {})
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
      (let [{:keys [type subtype]} (ex-data e)]
        (is (= error-type type))
        (is (= error-subtype subtype))))))

(defmacro throws
  [error-type error-subtype & body]
  `(throws-helper ~error-type ~error-subtype #(do ~@body)))

(defmacro sym-map
  "Builds a map from local symbols.
  Symbol names are turned into keywords and become the map's keys
  Symbol values become the map's values
  (let [a 1
        b 2]
    (sym-map a b))  =>  {:a 1 :b 2}"
  [& syms]
  (zipmap (map keyword syms) syms))

(defmacro inspect
  "Print some symbols for debugging, using pprint/cprint.
  (inspect foo bar) => foo:
                       {:a 1}
                       bar:
                       [:a :vector]"
  [& syms]
  (let [exprs (map #(vector (name %) %) syms)]
    `(inspect-helper ~@exprs)))

;;;;;;;;;;;;;;;;;;;; Macro-writing utils ;;;;;;;;;;;;;;;;;;;;

;; From: http://blog.nberger.com.ar/blog/2015/09/18/more-portable-complex-macro-musing/
(defn- cljs-env?
  "Take the &env from a macro, and return whether we are expanding into cljs."
  [env]
  (boolean (:ns env)))

(defmacro if-cljs
  "Return then if we are generating cljs code and else for Clojure code.
  https://groups.google.com/d/msg/clojurescript/iBY5HaQda4A/w1lAQi9_AwsJ"
  [then else]
  (if (cljs-env? &env) then else))


;;;;;;;;;;;;;;;;;;;; core.async utils ;;;;;;;;;;;;;;;;;;;;

(defn log-exception [e]
  ;; TODO: Enhance this to intelligently handle throw-far-error/ex-data
  (errorf (str "\nException:\n"
               #?(:clj (.getMessage ^Exception e) :cljs (.-message e))
               "\nStacktrace:\n"
               #?(:clj (join "\n" (map str
                                       (.getStackTrace ^Exception e)))
                  :cljs (.-stack e)))))

(defmacro go-safe [& body]
  `(if-cljs
    (cljs.core.async.macros/go
      (try
        ~@body
        (catch :default e#
          (log-exception e#)
          [:failure :exception-thrown])))
    (clojure.core.async/go
      (try
        ~@body
        (catch Exception e#
          (log-exception e#)
          [:failure :exception-thrown])))))

;;;;;;;;; Async test helper fns ;;;;;;;;;
;; Taken from
;; http://stackoverflow.com/questions/30766215/how-do-i-unit-test-clojure-core-async-go-macros/30781278#30781278

(defn test-async
  "Asynchronous test awaiting ch to produce a value or close."
  [ch]
  #?(:clj
     (<!! ch)
     :cljs
     (async done
            (take! ch (fn [_] (done))))))

(defn test-within-ms
  "Asserts that ch does not close or produce a value within ms. Returns a
  channel from which the value can be taken."
  [ms ch]
  (go (let [t (timeout ms)
            [v ch] (alts! [ch t])]
        (is (not= ch t)
            (str "Test should have finished within " ms "ms."))
        v)))

;;;;;;;;;;;;;;;;;;;; UUIDs ;;;;;;;;;;;;;;;;;;;;

(def UUIDLongMap
  {:high-long Long
   :low-long Long})

(defprotocol IUUID
  (to-hex-str [this] "Return the UUID as a 36-character canonical hex string.")
  (to-longs [this]   "Returns the UUID as two 64-bit longs.
                      High bits followed by low bits.")
  (to-long-map [this] "Returns the UUID a map with two keys, :high-long and
                       :low-long, whose values are 64-bit longs."))

(extend-protocol IUUID
  #?(:clj java.util.UUID :cljs cljs.core/UUID)
  (to-hex-str [this]
    (.toString this))

  (to-longs [this]
    #?(:clj [(.getMostSignificantBits this) (.getLeastSignificantBits this)]
       :cljs (let [hex-str (-> (.toString this)
                               (string/replace #"-" ""))
                   high-str (subs hex-str 0 16)
                   low-str (subs hex-str 16 32)]
               [(hex-str->long high-str) (hex-str->long low-str)])))

  (to-long-map [this]
    (let [[high-long low-long] (to-longs this)]
      (sym-map high-long low-long))))

(s/defn make-v1-uuid :- UUID
  []
  #?(:clj
     (clj-uuid/v1)
     :cljs
     ;; TODO: Replace this with a real v1 implementation
     (throw-far-error "V1 is not implemented in cljs yet"
                      :execution-error :not-implmented {})))

(s/defn make-v4-uuid :- UUID
  []
  #?(:cljs (random-uuid)
     :clj (clj-uuid/v4)))

(s/defn from-longs :- UUID
  [high-long :- Long
   low-long :- Long]
  #?(:clj
     (UUID. high-long low-long)
     :cljs
     (let [high-str (long->hex-str high-long)
           low-str (long->hex-str low-long)
           hex-str (string/join "-" [(subs high-str 0 8)
                                     (subs high-str 8 12)
                                     (subs high-str 12 16)
                                     (subs low-str 0 4)
                                     (subs low-str 4 16)])]
       (uuid hex-str))))

(s/defn from-long-map :- UUID
  [long-map :- UUIDLongMap]
  (from-longs (:high-long long-map) (:low-long long-map)))
