l(ns farbetter.utils
   (:require
    [#?(:clj clj-time.core :cljs cljs-time.core) :as t]
    [#?(:clj clj-time.format :cljs cljs-time.format) :as f]
    #?(:cljs [cljs-time.extend]) ;; to make = work as expected
    [#?(:clj clojure.core.async :cljs cljs.core.async)
     :refer [alts! take! timeout #?@(:clj [go <!!])]]
    [cognitect.transit :as transit]
    #?@(:clj
        [[clojure.test :refer [is]]
         [puget.printer :refer [cprint]]]
        :cljs
        [[cljs.pprint :refer [pprint]]
         [cljs.test :as test]]))
   #?(:cljs
      (:require-macros
       [cljs.core.async.macros :refer [go]]
       [cljs.test :refer [async is]]))
   #?(:clj
      (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
               [java.nio.charset StandardCharsets]
               [org.joda.time DateTime])
      :cljs
      (:import [goog.date Date UtcDateTime])))

(declare inspect-helper throws-helper)

;; TODO: Add schemas to these functions

;;;;;;;;;;; Exception handling and testing

(defn throw-far-error
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [description error-type error-subtype error-map]
  (let [error-map (merge error-map
                         {:type error-type :subtype error-subtype})]
    (throw (ex-info (str description ". error-map: " error-map)
                    error-map))))

(defmacro throws
  [error-type error-subtype & body]
  `(throws-helper ~error-type ~error-subtype #(do ~@body)))

(defn throws-helper
  [error-type error-subtype body]
  (try
    (body)
    (throw-far-error "Did not throw" :execution-error :did-not-throw {})
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
           (let [{:keys [type subtype]} (ex-data e)]
             (is (= error-type type))
             (is (= error-subtype subtype))))))

(defn abs
  "Returns the absolute value of n"
  [n]
  (if (neg? n)
    (- n)
    n))

(defn within
  "Asserts that the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin expected actual]
  (let [err (-> actual
                (- expected)
                abs)]
    (<= err error-margin)))

;;;;;;;;; Async test helper fns ;;;;;;;;;
;; Taken from
;; http://stackoverflow.com/questions/30766215/how-do-i-unit-test-clojure-core-async-go-macros/30781278#30781278

(defn test-async
  "Asynchronous test awaiting ch to produce a value or close."
  [ch]
  #?(:clj
     (<!! ch)
     :cljs
     (async done
            (take! ch (fn [_] (done))))))

(defn test-within-ms
  "Asserts that ch does not close or produce a value within ms. Returns a
  channel from which the value can be taken."
  [ms ch]
  (go (let [t (timeout ms)
            [v ch] (alts! [ch t])]
        (is (not= ch t)
            (str "Test should have finished within " ms "ms."))
        v)))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (defn construct-java-type-from-str [klass s]
     (.newInstance
      (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))
#?(:clj
   (defn construct-from-str [var-type var-value]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (defn get-env-var
     ([var-name var-type default-value]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))

     ([var-name var-type]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))))

;;;;;;;;;;; Time functions
(defn str->dt
  " Converts an ISO8601-formatted string to a date-time object"
  [s]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/parse formatter s)))

(defn dt->str
  "Converts a date-time object to an ISO8601-formatted string"
  [dt]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/unparse formatter dt)))

(defn add-days
  "Adds the specifed number of days to an ISO8601-formatted string
   Returns an ISO8601-formatted string"
  [dt-str days]
  (-> dt-str
      str->dt
      (t/plus (t/days days))
      dt->str))

(defn day-at
  "Takes a date-time, an hour, and a minute
   Returns a new date-tiem with the same date, but the
   specified hour and minute."
  ([dt] (day-at dt 0 0))
  ([dt h] (day-at dt h 0))
  ([dt h m]
   (t/date-time (t/year dt) (t/month dt) (t/day dt) h m)))

(defn get-num-midnights
  "Return the number of midnights (actually 23:59:59s) between two date-times"
  [begin end]
  (loop [night (t/date-time (t/year begin) (t/month begin) (t/day begin)
                            23 59 59)
         count 0]
    (if (t/within? begin end night)
      (recur (t/plus night (t/days 1)) (inc count))
      count)))

(defn dt+days
  "Takes a date-time, a number of days, and optionally hour and minute.
   Returns a new date-time offset by the number of days, with hour and
   minute set if they were provided."
  ([dt n]
   (dt+days dt n 0 0))
  ([dt n h]
   (dt+days dt n h 0))
  ([dt n h m]
   (day-at (t/plus dt (t/days n)) h m)))

(defn hrs-diff [dt1 dt2]
  (/ (t/in-seconds (t/interval dt1 dt2)) 3600.0))

(defn local->utc
  "Takes a local date-time, hours, and (optionally) minutes offset from UTC.
  Returns the corresponding date-time in UTC."
  ([dt offset-hrs]
   (local->utc dt offset-hrs 0))
  ([dt offset-hrs offset-mins]
   (t/minus dt (t/hours offset-hrs) (t/minutes offset-mins))))

(defn same-day? [dt1 dt2]
  (= [(t/year dt1) (t/month dt1) (t/day dt1)]
     [(t/year dt2) (t/month dt2) (t/day dt2)]))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;;;;;;;;;;;;;;;; Handy utilities

(defmacro sym-map
  "Builds a map from local symbols.
  Symbol names are turned into keywords and become the map's keys
  Symbol values become the map's values
  (let [a 1
        b 2]
    (sym-map a b))  =>  {:a 1 :b 2}"
  [& syms]
  (zipmap (map keyword syms) syms))

(defmacro inspect
  "Print some symbols for debugging, using pprint/cprint.
  (inspect foo bar) => foo:
                       {:a 1}
                       bar:
                       [:a :vector]"
  [& syms]
  (let [exprs (map #(vector (name %) %) syms)]
    `(inspect-helper ~@exprs)))

(defn inspect-helper [& exprs]
  (doseq [[expr-name expr-val] exprs]
    (println (str expr-name ":"))
    (#?(:clj cprint :cljs pprint) expr-val)))

;; dissoc-in function was stolen from:
;; https://github.com/clojure/core.incubator/blob/master/src
;; /main/clojure/clojure/core/incubator.clj
(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))


;;;;;;;;;;;;;;; Randomness

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))

;;;;;;;;; Serialization / Deserialization ;;;;;;;;;

(def initial-transit-buffer-size 4096)
(def transit-charset "UTF-8")
(def date-time-formatter (f/formatters :date-hour-minute-second-ms))
(def date-time-transit-tag "dt")

(def date-time-writer
  (transit/write-handler
   (constantly date-time-transit-tag)
   #(f/unparse date-time-formatter %)))

(def date-time-reader
  (transit/read-handler
   #(f/parse date-time-formatter %)))

(defn obj->transit [obj]
  #?(:clj
     (let [out (ByteArrayOutputStream. initial-transit-buffer-size)
           writer (transit/writer
                   out :json
                   {:handlers {DateTime date-time-writer}})]
       (transit/write writer obj)
       (.toString out transit-charset))
     :cljs
     (transit/write (transit/writer
                     :json
                     {:handlers {UtcDateTime date-time-writer}})
                    obj)))

(defn transit->obj [^String transit-str]
  (when transit-str
    #?(:clj
       (let [bytes (.getBytes transit-str transit-charset)
             in (ByteArrayInputStream. bytes)
             reader (transit/reader
                     in :json
                     {:handlers {date-time-transit-tag date-time-reader}})]
         (transit/read reader))
       :cljs
       (transit/read (transit/reader
                      :json
                      {:handlers {date-time-transit-tag date-time-reader}})
                     transit-str))))
