(ns farbetter.utils
  (:require
   #?(:cljs [cljs.test :as t])
   #?@(:clj
       [[clojure.test :refer [is]]
        [puget.printer :refer [cprint]]]
       :cljs
       [[cljs.pprint :refer [pprint]]]))
  #?(:cljs (:require-macros
            [cljs.test :refer [is]])))

(declare inspect-helper throws-helper)

;;;;;;;;;;; Exception handling and testing

(defn throw-far-error
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [description error-type error-subtype error-map]
  (let [error-map (merge error-map
                         {:type error-type :subtype error-subtype})]
    (throw (ex-info (str description ". error-map: " error-map)
                    error-map))))

(defmacro throws
  [error-type error-subtype & body]
  `(throws-helper ~error-type ~error-subtype #(do ~@body)))

(defn throws-helper
  [error-type error-subtype body]
  (try
    (body)
    (throw-far-error "Did not throw" :execution-error :did-not-throw {})
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
           (let [{:keys [type subtype]} (ex-data e)]
             (is (= error-type type))
             (is (= error-subtype subtype))))))

(defn abs
  "Returns the absolute value of n"
  [n]
  (if (neg? n)
    (- n)
    n))

(defn within
  "Asserts that the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin expected actual]
  (let [err (-> actual
                (- expected)
                abs)]
    (<= err error-margin)))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (defn construct-java-type-from-str [klass s]
     (.newInstance
      (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))
#?(:clj
   (defn construct-from-str [var-type var-value]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (defn get-env-var
     ([var-name var-type default-value]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))

     ([var-name var-type]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;;;;;;;;;;;;;;;; Handy utilities

(defmacro sym-map
  "Builds a map from local symbols.
  Symbol names are turned into keywords and become the map's keys
  Symbol values become the map's values
  (let [a 1
        b 2]
    (sym-map a b))  =>  {:a 1 :b 2}"
  [& syms]
  (zipmap (map keyword syms) syms))

(defmacro inspect
  "Print some symbols for debugging, using pprint/cprint.
  (inspect foo bar) => foo:
                       {:a 1}
                       bar:
                       [:a :vector]"
  [& syms]
  (let [exprs (map #(vector (name %) %) syms)]
    `(inspect-helper ~@exprs)))

(defn inspect-helper [& exprs]
  (doseq [[expr-name expr-val] exprs]
    (println (str expr-name ":"))
    (#?(:clj cprint :cljs pprint) expr-val)))

;; dissoc-in function was stolen from:
;; https://github.com/clojure/core.incubator/blob/master/src
;; /main/clojure/clojure/core/incubator.clj
(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))


;;;;;;;;;;;;;;; Randomness

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))
