(ns farbetter.utils)

;;;;;;;;;;; cljc helpers

(defn- cljs-env?
  "Take the &env from a macro, and tell whether we are expanding into cljs."
  [env]
  (boolean (:ns env)))

(defmacro if-cljs
  "Return then if we are generating cljs code and else for Clojure code.
   https://groups.google.com/d/msg/clojurescript/iBY5HaQda4A/w1lAQi9_AwsJ"
  [then else]
  (if (cljs-env? &env)
    then
    else))

;;;;;;;;;;; Exception handling and testing

(defmacro throw-far-error
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [description error-type error-subtype error-map]
  `(let [emap# (merge ~error-map
                      {:type ~error-type :subtype ~error-subtype})]
     (throw (ex-info (str ~description ". error-map: " emap#)
                     emap#))))

(defmacro is [& args]
  `(if-cljs
    (cljs.test/is ~@args)
    (clojure.test/is ~@args)))

(defmacro throws
  "Asserts that the body throws a particular far-error.
  Must be used inside a deftest block."
  [error-type error-subtype & body]
  `(try
     ~@body
     (throw-far-error "Did not throw" :did-not-throw :did-not-throw {})
     (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e#
            (let [data# (ex-data e#)]
              (is (= ~error-type (:type data#)))
              (is (= ~error-subtype (:subtype data#)))))))

(defn abs
  "Returns the absolute value of n"
  [n]
  (if (neg? n)
    (- n)
    n))

(defn within
  "Asserts that the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin expected actual]
  (let [err (-> actual
                (- expected)
                abs)]
    (<= err error-margin)))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (defn construct-java-type-from-str [klass s]
     (.newInstance
      (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))
#?(:clj
   (defn construct-from-str [var-type var-value]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (defn get-env-var
     ([var-name var-type default-value]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))

     ([var-name var-type]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;; Builds a map from local symbols.
;; Symbol names are turned into keywords and become the map's keys
;; Symbol values become the map's values
;; (let [a 1
;;       b 2]
;;   (sym-map a b))  => {:a 1 :b 2}
(defmacro sym-map [& syms]
  (zipmap (map keyword syms) syms))

;; Print some symbols for debugging, using cprint
;; (inspect foo bar) =>
;;                      foo:
;;                      {:a 1}
;;                      bar:
;;                      [:a :vector]
(defmacro inspect [& syms]
  (let [exprs (map (fn [sym]
                     `(do
                        (println ~(str (name sym) ":"))
                        (puget.printer/cprint ~sym)))
                   syms)]
    `(do (require 'puget.printer) ~@exprs)))


;;;;;;;;;;;;;;; Randomness

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))
