(ns farbetter.utils
  (:require
   [clojure.math.numeric-tower :as math]
   [clojure.test :as test]))

;;;;;;;;;;; Exception handling and testing

(defmacro throw-far-error [description error-type error-subtype error-map]
  `(let [emap# (merge ~error-map
                      {:type ~error-type :subtype ~error-subtype})]
     (throw (ex-info (str ~description ". error-map: " emap#)
                     emap#))))

;; Asserts that the body throws a particular far-error
;; Must be used inside a deftest block
(defmacro throws [error-type error-subtype & body]
  `(try
     ~@body
     (throw-far-error "Did not throw" :did-not-throw :did-not-throw {})
     (catch clojure.lang.ExceptionInfo e#
       (let [data# (ex-data e#)]
         (test/is (= ~error-type (:type data#)))
         (test/is (= ~error-subtype (:subtype data#)))))))

;; Asserts that the actual value is within error-margin of the
;; expected value.
;; Actual and expected must be numeric
(defn within [error-margin expected actual]
  (let [err (-> actual
                (- expected)
                (math/abs))]
    (<= err error-margin)))

;;;;;;;;;;; Environment variables

(defn construct-java-type-from-str [klass s]
  (.newInstance
   (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
   (object-array [s])))

(defn construct-from-str [var-type var-value]
  (if (= clojure.lang.Keyword var-type)
    (keyword var-value)
    (construct-java-type-from-str var-type var-value)))

(defn get-env-var
  ([var-name var-type default-value]
   (if-let [var-value-str (System/getenv var-name)]
     (construct-from-str var-type var-value-str)
     default-value))

  ([var-name var-type]
   (if-let  [var-value (get-env-var var-name var-type nil)]
     var-value
     (throw-far-error "Environment variable not found"
                      :environment-error :env-var-not-found
                      {:var-name var-name :var-type var-type}))))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959]
    (* earth-radius-mi
       (Math/acos (+ (* (Math/sin (Math/toRadians lat1))
                        (Math/sin (Math/toRadians lat2)))
                     (* (Math/cos (Math/toRadians lat1))
                        (Math/cos (Math/toRadians lat2))
                        (Math/cos (- (Math/toRadians long1)
                                     (Math/toRadians long2)))))))))

;; Builds a map from local symbols.
;; Symbol names are turned into keywords and become the map's keys
;; Symbol values become the map's values
;; e.g.: (let [a 1
;;             b 2]
;;         (sym-map a b))  => {:a 1 :b 2}
(defmacro sym-map [& syms]
  (zipmap (map keyword syms) syms))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn xmap
  "Transform a map's values by setting them to (f key)"
  [m f]
  (reduce #(assoc %1 %2 (f %2)) m (keys m)))

(defn keys-of-diff-vals
  "Returns a seq of keys whose value in the first map differs from the
  value in the second map. Keys that only exist in the second map
  are ignored."
  [m1 m2]
  (keep identity
        (for [[k1 v1] m1]
          (when (not= v1 (m2 k1))
            k1))))

(defn indexes-of-diff-vals
  "Returns a seq of indexes whose value in the first vector differs from
  the value in the second vector. Vectors should be the same length."
  [v1 v2]
  (keep identity
        (for [i (range (count v1))]
          (when (not= (v1 i) (v2 i))
            i))))
