(ns farbetter.tube
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   #?(:clj [gniazdo.core :as ws])
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])
     :clj
     (:import [java.util Arrays])))

(declare make-websocket)

(def wait-ms 5)
(def ws-url-pattern (re-pattern "^(ws|wss)://.+"))
(def valid-states #{:start :connected :closed})

(defprotocol ITube
  (close [this] "Close the tube.")
  (get-state [this] "Get the tube's state.")
  (-start-send-block [this] "Internal use only.")
  (-start-connect-timeout-block [this] "Internal use only.")
  (-start-close-chan-watch-block [this] "Internal use only."))

(defrecord Tube [to-internet-chan from-internet-chan ws-closer ws-sender
                 tube-closer state-atom close-chan
                 connect-timeout-ms connection-chan on-error on-disconnect]
  ITube
  (close [this]
    (ws-closer)
    (tube-closer))

  (get-state [this]
    @state-atom)

  (-start-send-block [this]
    (go-safe
     (loop [data nil]
       (case @state-atom
         :closed nil
         :start (do
                  (ca/<! (ca/timeout wait-ms))
                  (recur nil))
         :connected (do
                      (when data
                        (ws-sender data))
                      (if-let [data (ca/<! to-internet-chan)]
                        (recur data)
                        (close this)))))))

  (-start-connect-timeout-block [this]
    (go-safe
     (let [end-time (+ (u/get-current-time-ms) connect-timeout-ms)]
       (loop []
         (let [timeout-chan (ca/timeout wait-ms)
               [val ch] (ca/alts! [connection-chan timeout-chan]
                                  :priority true)]
           (when (and (= timeout-chan ch)
                      (= :start @state-atom))
             (if (< (u/get-current-time-ms) end-time)
               (recur)
               (let [reason :timeout-on-connect]
                 (tube-closer)
                 (on-error reason)
                 (on-disconnect reason)))))))))

  (-start-close-chan-watch-block [this]
    (go-safe
     (when (ca/<! close-chan)
       (close this)))))

(defn make-tube
  "Make a websocket connection.
   Arguments:
   - url - URL of the websocket server. Required.
   - to-internet-chan - Channel for sending data to the
        internet. Required.
   - from-internet-chan - Channel for receiving data from the
        internet. Required.
   - opts - Optional map of options. Supported keys:
        - :on-connect - Callback of no arguments. Will be called when the
             websocket connects with the remote peer. Optional.
        - :on-disconnect - Callback of one argument (reason). Will be called
             when the websocket disconnects. Optional.
        - :on-error - Callback of one argument (error). Will be called on any
             errors encountered by the websocket. Optional.
        - :connect-timeout-ms - Milliseconds to wait before timing
             out on connect. Optional. Defaults to 10000.
        - :ws-factory - Used for testing only. Callback of six args:
             [url on-connect on-disconnect on-error on-rcv tube-closer].
             Must return a sequence of [ws-closer ws-sender]. Optional"
  ([url to-internet-chan from-internet-chan]
   (make-tube url to-internet-chan from-internet-chan {}))
  ([url to-internet-chan from-internet-chan opts]
   (when-not (re-find ws-url-pattern url)
     (throw-far-error "URL must begin with ws:// or wss://"
                      :illegal-argument :illegal-url (sym-map url)))
   (let [{:keys [on-connect on-disconnect on-error connect-timeout-ms
                 ws-factory]
          :or {connect-timeout-ms 10000
               ws-factory make-websocket}} opts
         state-atom (atom :start)
         valid-state? (fn [new-state]
                        (contains? valid-states new-state))
         _ (set-validator! state-atom valid-state?)
         close-chan (ca/chan 1)
         connection-chan (ca/chan 1)
         on-rcv* (fn [data]
                   (debugf "Got data: %s" data)
                   (when-not (ca/put! from-internet-chan data)
                     (ca/put! close-chan true)))
         on-connect* (fn []
                       (infof "Websocket connected to %s" url)
                       (reset! state-atom :connected)
                       (ca/put! connection-chan true)
                       (when on-connect
                         (on-connect)))
         tube-closer #(when-not (= :closed @state-atom)
                        (infof "Closing tube to %s" url)
                        (ca/close! from-internet-chan)
                        (ca/close! to-internet-chan)
                        (reset! state-atom :closed))
         on-disconnect* (fn [reason]
                          (tube-closer)
                          (when on-disconnect
                            (on-disconnect reason)))
         on-error* (fn [error]
                     (when on-error
                       (on-error error)))
         [ws-closer ws-sender] (ws-factory url on-connect* on-disconnect*
                                           on-error* on-rcv* tube-closer)
         args (sym-map to-internet-chan ws-closer ws-sender tube-closer
                       state-atom connect-timeout-ms connection-chan close-chan)
         tube (map->Tube (assoc args
                                :on-error on-error*
                                :on-disconnect on-disconnect*))]
     (when (and ws-closer ws-sender)
       (-start-send-block tube)
       (-start-connect-timeout-block tube)
       (-start-close-chan-watch-block tube)
       tube))))

#?(:clj
   (defn make-websocket-clj
     [url on-connect on-disconnect on-error on-rcv tube-closer]
     (let [on-bin (fn [bytes offset length]
                    (on-rcv (Arrays/copyOfRange bytes offset
                                                (+ offset length))))
           socket (ws/connect url
                              :on-connect (fn [session]
                                            (on-connect))
                              :on-close (fn [status reason]
                                          (on-disconnect reason))
                              :on-error on-error
                              :on-receive on-rcv
                              :on-binary on-bin)
           closer #(ws/close socket)
           sender (fn [data]
                    ;; Send-msg mutates binary data, so we make a
                    ;; copy if data is binary
                    (if (string? data)
                      (ws/send-msg socket data)
                      (ws/send-msg socket (Arrays/copyOf data
                                                         (count data)))))]
       [closer sender])))

#?(:cljs
   (defn make-websocket-node
     [url on-connect on-disconnect on-error on-rcv tube-closer]
     (let [WSC (goog.object.get (js/require "websocket") "client")
           client (WSC.)
           conn-atom (atom nil)
           msg-handler (fn [msg-obj]
                         (let [type (goog.object.get msg-obj "type")
                               data (if (= "utf8" type)
                                      (goog.object.get msg-obj "utf8Data")
                                      (-> (goog.object.get msg-obj "binaryData")
                                          (js/Int8Array.)))]
                           (on-rcv data)))
           conn-handler (fn [conn]
                          (reset! conn-atom conn)
                          (on-connect)
                          (.on conn "close" (fn [reason description]
                                              (on-disconnect description)))
                          (.on conn "message" msg-handler)
                          (.on conn "error" on-error))
           closer #(if @conn-atom
                     (.close @conn-atom)
                     (.abort client))
           sender (fn [data]
                    (if (string? data)
                      (.sendUTF @conn-atom data)
                      (.sendBytes @conn-atom (js/Buffer. data))))
           failure-handler  (fn [err]
                              (let [reason [:connect-failure err]]
                                (tube-closer)
                                (on-error reason)
                                (on-disconnect reason)))]
       (.on client "connectFailed" failure-handler)
       (.on client "connect" conn-handler)
       (.connect client url)
       [closer sender])))

#?(:cljs
   (defn make-websocket-browser
     ;; TODO: Test and fully implement
     [url on-connect on-disconnect on-error on-rcv tube-closer]
     (let [socket (js/WebSocket. url)
           closer #(.close socket)
           sender #(.send socket %)]
       (.onopen socket on-connect)
       (.onclose socket (fn [] (on-disconnect :closed)))
       (.onerror socket on-error)
       (.onmessage socket #(on-rcv (goog.object.get % "data")))
       (set! (.-binaryType socket) "arraybuffer")
       [closer sender])))

#?(:cljs
   (defn make-websocket-ios
     ;; TODO: Test and fully implement
     [url on-connect on-disconnect on-error on-rcv tube-closer]
     (let [socket (.makeWithCallbacks js/FarWebSocket url on-connect
                                      on-disconnect on-error on-rcv)
           closer #(.close socket)
           sender #(.send socket %)]
       [closer sender])))

(defn- make-websocket
  "Should return a sequence of [close-fn send-fn]"
  [url on-connect on-disconnect on-error on-rcv tube-closer]
  (try
    (let [platform (u/get-platform-kw)
          factory #?(:clj make-websocket-clj
                     :cljs (case platform
                             :node make-websocket-node
                             :ios make-websocket-ios
                             :browser make-websocket-browser))]
      (factory url on-connect on-disconnect on-error on-rcv tube-closer))
    (catch #?(:clj Exception :cljs :default) e
      (let [reason [:construction-failed e]]
        (tube-closer)
        (on-error reason)
        (on-disconnect reason))
      nil)))
