(ns farbetter.string-store.impl
  (:require
   [clojure.core.async :as ca]
   [farbetter.string-store.storage :as storage]
   [farbetter.roe :as roe]
   [farbetter.utils :as u :refer [sym-map]]
   [schema.core :as s]
   [taoensso.timbre :as timbre :refer [debugf errorf infof tracef warnf]])
  #?(:clj
     (:import
      [java.nio ByteBuffer])))

(defn make-row-id [name version]
  (str name "#" version))

(defn get-chunk [storage index chunk-id]
  (u/go-sf
   (let [row (u/call-sf! storage/get-row storage chunk-id)
         {:keys [chunk]} row
         chunk #?(:clj (if (instance? ByteBuffer chunk)
                         (.array chunk)
                         chunk)
                  :cljs chunk)]
     [index chunk])))

(defn get-chunks [storage chunk-ids]
  (u/go-sf
   (let [chans (map-indexed (partial get-chunk storage)
                            chunk-ids)
         result-chan (ca/merge chans)
         result (atom (sorted-map))]
     (dotimes [n (count chunk-ids)]
       (let [[status ret] (ca/<! result-chan)]
         (if (= :success status)
           (let [[index chunk] ret]
             (swap! result assoc index chunk))
           (let [reason ret]
             (if (instance? #?(:clj Throwable :cljs js/Error) reason)
               (throw reason)
               (u/throw-far-error
                "get-chunk failed"
                :execution-error :search-failure
                (sym-map status reason)))))))
     (vals @result))))

(defn load-string* [storage name version]
  (u/go-sf
   (let [version (or version
                     (u/call-sf! storage/get-version storage name))]
     (when version
       (let [header-row-id (make-row-id name version)
             header-row (u/call-sf! storage/get-row storage header-row-id)]
         (when (seq header-row)
           (let [{:keys [chunk-ids]} header-row
                 chunks (u/call-sf! get-chunks storage chunk-ids)
                 deflated-bytes (u/concat-byte-arrays chunks)
                 inflated-bytes (u/inflate deflated-bytes)]
             (roe/avro-byte-array->edn :string :string inflated-bytes))))))))

(defn byte-array->chunks [chunk-size ba]
  (loop [offset 0
         output []]
    (if (>= offset (count ba))
      output
      (let [end-offset (+ offset chunk-size)
            chunk (u/slice ba offset end-offset)]
        (recur end-offset
               (conj output chunk))))))

(defn make-uuid-str []
  (-> (u/make-v4-uuid)
      (u/uuid->hex-str)))

(defn store-chunks [storage name version chunks]
  (u/go-sf
   (let [chunk-ids (mapv (fn [_]
                           (make-uuid-str))
                         chunks)
         timestamp (u/get-current-time-ms)
         header-row-id (make-row-id name version)
         header-row (sym-map timestamp chunk-ids)]
     (u/call-sf! storage/put-row storage header-row-id header-row)
     (doseq [[id chunk] (map vector chunk-ids chunks)]
       (u/call-sf! storage/put-row storage id (sym-map chunk))))))

(defn store-string* [storage name version s]
  (u/go-sf
   (let [version (or version
                     (u/call-sf! storage/inc-version storage name))
         inflated-bytes (roe/edn->avro-byte-array :string s)
         deflated-bytes (u/deflate inflated-bytes)
         chunk-size (u/call-sf! storage/get-chunk-size storage)
         chunks (byte-array->chunks chunk-size deflated-bytes)]
     (u/call-sf! store-chunks storage name version chunks)
     version)))
