(ns farbetter.string-store.ddb-storage
  (:require
   [clojure.core.async :as ca]
   [farbetter.string-store.storage :as storage]
   [farbetter.utils :as u :refer [sym-map]]
   [farbetter.utils.attempt-strategies :as strats]
   [schema.core :as s]
   [taoensso.timbre :as timbre
    :refer [debugf error errorf infof]])
  (:import
   [com.amazonaws.handlers AsyncHandler]
   [com.amazonaws.services.dynamodbv2 AmazonDynamoDBAsyncClient]
   [com.amazonaws.services.dynamodbv2.model AttributeValue AttributeValueUpdate
    GetItemRequest]
   [java.nio ByteBuffer]))

(def version-row-tag "#VERSION_ROW")
(def retry-strategy (strats/max-retries
                     4
                     (strats/randomize-strategy
                      0.2
                      (strats/multiplicative-strategy 100 2))))

(def timeout-strategy (strats/multiplicative-strategy 1000 2))

(defn- parse-number
  [^String ns]
  (if (.contains ns ".")
    (Double/parseDouble ns)
    (Long/parseLong ns)))

(defn get-value [av]
  (let [[type val] (some #(when (not (nil? (val %))) %)
                         (dissoc (bean av) :class))]
    (case type
      (:s :b :BOOL) val
      (:SS :BS) (into #{} val)
      :n (parse-number val)
      :NS (into #{} (map parse-number val))
      :l (into [] (map get-value val))
      :m (into {} (map (fn [[k v]] [k (get-value v)]) val))
      :NULL nil)))

(defn attr-map->clj-map [am]
  (reduce-kv (fn [acc k av]
               (assoc acc (keyword k)
                      (get-value av)))
             {} (into {} am)))

(defn clj-key->str-key [k]
  (if (keyword? k)
    (name k)
    (str k)))

(defn clj-value->attr-value [value]
  (let [attr (AttributeValue.)]
    (cond
      (nil? value) (.setNULL attr true)
      (number? value) (.setN attr (str value))
      (string? value) (.setS attr value)
      (instance? Boolean value) (.setBOOL attr value)
      (sequential? value) (.setL attr
                                 (map clj-value->attr-value value))
      (u/byte-array? value) (.setB attr (ByteBuffer/wrap value))
      (instance? ByteBuffer value) (.setB attr value)
      :else (u/throw-far-error
             (str "Value of type " (class value)
                  " is not supported")
             :execution-error :unsupported-ddb-type
             (sym-map value)))
    attr))

(defn clj-map->attr-map [m]
  (reduce-kv (fn [acc k v]
               (assoc acc (clj-key->str-key k)
                      (clj-value->attr-value v)))
             {} m))

(defn make-row-id [user-id-str k]
  (str user-id-str "#" k))

(defn make-handler [ret-chan xf-result]
  (reify AsyncHandler
    (onError [this e]
      (ca/put! ret-chan [:failure e]))
    (onSuccess [this rq result]
      (ca/put! ret-chan [:success
                         (try
                           (xf-result result)
                           (catch Exception e
                             [:failure e]))]))))

(defn get-row* [storage k]
  (let [{:keys [user-id-str client table-name]} storage
        row-id (make-row-id user-id-str k)
        attr-value (AttributeValue. row-id)
        consistent? true
        ret-chan (ca/chan)
        xf-result (fn [ret]
                    (->> (.getItem ret)
                         (attr-map->clj-map)))
        handler (make-handler ret-chan xf-result)
        ret-future (.getItemAsync client table-name
                                  {"id" attr-value} consistent?
                                  handler)]
    ret-chan))

(defn put-row* [storage k v]
  (let [{:keys [user-id-str client table-name]} storage
        row-id (make-row-id user-id-str k)
        row (clj-map->attr-map (assoc v :id row-id))
        ret-chan (ca/chan)
        xf-result (constantly nil)
        handler (make-handler ret-chan xf-result)]
    (.putItemAsync client table-name row handler)
    ret-chan))

(defn inc-version* [storage k]
  (let [{:keys [user-id-str client table-name]} storage
        row-id (str (make-row-id user-id-str k) version-row-tag)
        attr-value (AttributeValue. row-id)
        ret-chan (ca/chan)
        xf-result #(-> (.getAttributes %)
                       (attr-map->clj-map)
                       (:version))
        handler (make-handler ret-chan xf-result)
        av-update (AttributeValueUpdate.)]
    (.setValue av-update (clj-value->attr-value 1))
    (.setAction av-update "ADD")
    (.updateItemAsync client table-name {"id" attr-value}
                      {"version" av-update}
                      "ALL_NEW" handler)
    ret-chan))

(defn get-version* [storage k]
  (let [xf (map #(let [[status ret] %]
                   (if (= :success status)
                     [:success (:version ret)]
                     %)))
        version-chan (ca/chan 1 xf)]
    (ca/pipe (get-row* storage (str k version-row-tag))
             version-chan)
    version-chan))


(defrecord DDBStorage [client user-id-str table-name]
  storage/IStorage
  (inc-version [this k]
    (u/attempt-w-retry-and-timeout
     #(inc-version* this k)
     retry-strategy timeout-strategy))

  (get-version [this k]
    (u/attempt-w-retry-and-timeout
     #(get-version* this k)
     retry-strategy timeout-strategy))

  (get-row [this k]
    (u/attempt-w-retry-and-timeout
     #(get-row* this k)
     retry-strategy timeout-strategy))

  (put-row [this k v]
    (u/attempt-w-retry-and-timeout
     #(put-row* this k v)
     retry-strategy timeout-strategy))

  (get-chunk-size [this]
    (u/go-sf
     (- (* 1024 400)
        100))))

(defn make-ddb-storage [endpoint user-id-str table-name]
  (let [client (AmazonDynamoDBAsyncClient.)
        _ (.setEndpoint client endpoint)]
    (->DDBStorage client user-id-str table-name)))
