(ns farbetter.roe.utils
  (:require
   #?(:cljs [cljsjs.bytebuffer])
   [farbetter.utils :as u
    :refer [byte-array? throw-far-error
            #?@(:cljs [byte-array])
            #?@(:clj [inspect sym-map])]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(declare equivalent?)

#?(:cljs (def Exception js/Error))
#?(:cljs (def class type))
#?(:cljs (def ByteBuffer js/ByteBuffer))

;; Hack to get around js numbers all being floating point
#?(:cljs
   (defn float? [x]
     (and (number? x)
          (not (integer? x)))))

(defn bytes->utf8-string [bs]
  #?(:clj
     (String. #^bytes bs "UTF-8")
     :cljs
     (let [num-bytes (count bs)
           buf (.wrap ByteBuffer (.-buffer bs))]
       (.readUTF8String buf num-bytes (.-METRICS_BYTES ByteBuffer)))))

(defn utf8-str->byte-array [s]
  #?(:clj
     (.getBytes ^String s "UTF-8")
     :cljs
     (let [bb (.fromUTF8 ByteBuffer s)]
       (-> (.toArrayBuffer bb)
           (js/Int8Array.)))))

(defn get-avro-type [schema]
  (cond
    (sequential? schema) :union
    (map? schema) (:type schema)
    :else schema))

(defn make-field-name->field-type-map [schema]
  (let [f (fn [acc {:keys [name type]}]
            (assoc acc name type))]
    (reduce f {} (:fields schema))))

(defn equivalent-records? [schema-a a schema-b b]
  (let [a-keys (set (keys a))
        b-keys (set (keys b))
        a-field-type-map (make-field-name->field-type-map schema-a)
        b-field-type-map (make-field-name->field-type-map schema-b)]
    (and (= a-keys b-keys)
         (reduce (fn [acc field-name]
                   (let [a-type (a-field-type-map field-name)
                         b-type (b-field-type-map field-name)]
                     (and acc
                          (equivalent? a-type (a field-name)
                                       b-type (b field-name)))))
                 true a-keys))))

(defn equivalent-arrays? [schema a b]
  (let [items-schema (:items schema)]
    (and (= (count a) (count b))
         (reduce (fn [acc i]
                   (and acc
                        (equivalent? items-schema (a i)
                                     items-schema (b i))))
                 true (range (count a))))))

(defn equivalent-maps? [schema a b]
  (let [values-schema (:values schema)]
    (and (= (set (keys a)) (set (keys b)))
         (reduce-kv (fn [acc k v]
                      (and acc
                           (equivalent? values-schema (a k)
                                        values-schema (b k))))
                    true a))))

(defn equivalent-unions? [union-schema a b]
  (let [preds (map (fn [candidate-schema]
                     (fn [[a b]]
                       (try
                         (equivalent? candidate-schema a candidate-schema b)
                         (catch Exception e
                           false))))
                   union-schema)
        result ((apply some-fn preds) [a b])]
    result))

(defn equivalent-bytes-or-strings? [a b]
  (cond
    (and (byte-array? a)
         (byte-array? b))
    (u/equivalent-byte-arrays? a b)

    (and (string? a)
         (string? b))
    (= a b)

    (and (string? a)
         (byte-array? b))
    (let [b-str (bytes->utf8-string b)
          result (= a b-str)]
      result)

    (and (string? b)
         (byte-array? a))
    (= b (bytes->utf8-string a))

    :else (throw-far-error "Incompatible types"
                           :illegal-argument :incompatible-types
                           (sym-map a b))))

(defn equivalent-numbers? [type-a a type-b b]
  (let [sum (fn [x y]
              #?(:clj (+ x y)
                 :cljs
                 (if (u/long? x)
                   (.add x y)
                   (if (u/long? y)
                     (.add y x)
                     (+ x y)))))
        err-margin #(/ (u/abs (sum a b)) 1e5)
        float-zero? #(let [n (if (u/long? %)
                               (u/long->float %)
                               %)]
                       (< n 1e-45))
        ->float #(if (u/long? %)
                   (u/long->float %)
                   (float %))
        ->double #(if (u/long? %)
                    (u/long->float %)
                    (double %))]
    (cond
      (or (= :float type-a)
          (= :float type-b))
      (or (and (float-zero? a) (float-zero? b))
          (u/within? (err-margin) (->float a) (->float b)))

      (or (= :double type-a)
          (= :double type-b))
      (or (and (float-zero? a) (float-zero? b))
          (u/within? (err-margin) (->double a) (->double b)))

      (u/long? a)
      (u/long= a b)

      (u/long? b)
      (u/long= b a)

      :else
      (= a b))))

(defn equivalent? [schema-a a schema-b b]
  (when (or (nil? schema-a)
            (nil? schema-b))
    (throw-far-error "Schema is nil"
                     :illegal-schema :schema-is-nil
                     (sym-map a b schema-a schema-b)))
  (let [type-a (get-avro-type schema-a)
        type-b (get-avro-type schema-b)]
    (case type-a
      :bytes (equivalent-bytes-or-strings? a b)
      :string (equivalent-bytes-or-strings? a b)
      :float (equivalent-numbers? type-a a type-b b)
      :double (equivalent-numbers? type-a a type-b b)
      :int (equivalent-numbers? type-a a type-b b)
      :long (equivalent-numbers? type-a a type-b b)
      :record (equivalent-records? schema-a a schema-b b)
      :array (equivalent-arrays? schema-a a b)
      :map (equivalent-maps? schema-a a b)
      :fixed (u/equivalent-byte-arrays? a b)
      :union (equivalent-unions? schema-a a b)
      (= a b))))
