(ns farbetter.roe.schemas
  (:require
   [clojure.set :refer [union]]
   [farbetter.utils :as u
    :refer [byte-array? long? throw-far-error
            #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true] ;; :include-macros is for cljs
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare AvroSchema AvroData)

(def avro-name-pattern-str "[A-Za-z-][A-Za-z0-9-.]*")
(def avro-name-pattern (re-pattern avro-name-pattern-str))

;; Note: These fns must not put elsewhere and forward declared using
;; (declare ...) because Prismatic schema then can't resolve
;; them at macro-expansion time and schema checking throws vague
;; exceptions

(defn valid-avro-name? [n]
  (and (keyword? n)
       (->> (name n)
            (re-matches avro-name-pattern)
            (boolean))))

;; We are loose on AvroNames here so that the manual error checking
;; can give the correct error.
(def AvroName s/Keyword)
(def AvroEnumSymbol s/Keyword)
(def avro-primitive-types
  #{:null :boolean :int :long :float :double :bytes :string})
(def avro-primitive-type-names
  #{"null" "boolean" "int" "long" "float" "double" "bytes" "string"})
(def avro-named-types #{:record :fixed :enum})
(def avro-recursive-types #{:record :array :map :union})
(def avro-built-in-types (union avro-primitive-types avro-named-types
                                avro-recursive-types))

(def AvroPrimitiveType
  (apply s/enum avro-primitive-types))

(def AvroPrimitiveMapType
  {(s/required-key :type) AvroPrimitiveType})

(def AvroSortOrder
  (s/enum :ascending :descending :ignore))

(def AvroFieldSchema
  {(s/required-key :type) (s/recursive #'AvroSchema)
   (s/required-key :name) AvroName
   (s/required-key :default) s/Any
   (s/optional-key :doc) s/Str
   (s/optional-key :aliases) [AvroName]
   (s/optional-key :order) AvroSortOrder}) ;; :order is currently ignored

(def AvroRecordSchema
  {(s/required-key :type) (s/eq :record)
   (s/required-key :name) AvroName
   (s/required-key :fields) [AvroFieldSchema]
   (s/optional-key :doc) s/Str
   (s/optional-key :aliases) [AvroName]})

(def AvroEnumSchema
  {(s/required-key :type) (s/eq :enum)
   (s/required-key :name) AvroName
   (s/required-key :symbols) #{AvroEnumSymbol}
   (s/optional-key :doc) s/Str
   (s/optional-key :aliases) [AvroName]})

(def AvroFixedSchema
  {(s/required-key :type) (s/eq :fixed)
   (s/required-key :name) AvroName
   (s/required-key :size) s/Int
   (s/optional-key :doc) s/Str
   (s/optional-key :aliases) [AvroName]})

(def AvroArraySchema
  {(s/required-key :type) (s/eq :array)
   (s/required-key :items) (s/recursive #'AvroSchema)})

(def AvroMapSchema
  {(s/required-key :type) (s/eq :map)
   (s/required-key :values) (s/recursive #'AvroSchema)})

(def AvroUnionSchema [(s/recursive #'AvroSchema)])

(def AvroSchema
  (s/conditional
   vector? AvroUnionSchema
   #(contains? avro-primitive-types %) AvroPrimitiveType
   #(contains? avro-primitive-types (:type %)) AvroPrimitiveMapType
   #(= :record (:type %)) AvroRecordSchema
   #(= :enum (:type %)) AvroEnumSchema
   #(= :array (:type %)) AvroArraySchema
   #(= :map (:type %)) AvroMapSchema
   #(= :fixed (:type %)) AvroFixedSchema
   keyword? AvroName))

(def AvroNameOrString (s/conditional
                       string? s/Str
                       keyword? AvroName))

(def AvroRecordOrMap {AvroNameOrString (s/recursive #'AvroData)})
(def AvroEnum AvroName)
(def AvroNull (s/eq nil))
(def AvroArray [(s/recursive #'AvroData)])
(def AvroFixedOrBytes (s/pred byte-array?))
(def AvroBoolean (s/enum true false))
(def AvroLong (s/pred long?))
(def AvroString s/Str)

(def AvroNum
  (s/maybe
   (s/conditional
    long? AvroLong
    number? s/Num)))

(def AvroData
  (s/maybe
   (s/conditional
    #(or (true? %) (false? %)) AvroBoolean
    #(or (long? %) (number? %)) AvroNum
    byte-array? AvroFixedOrBytes
    map? AvroRecordOrMap
    keyword? AvroEnum
    sequential? AvroArray
    string? AvroString)))
