(ns farbetter.roe.io-streams
  (:require
   [cljsjs.bytebuffer]
   [farbetter.utils :as u :refer [throw-far-error]]
   [goog.object])
  (:require-macros
   [farbetter.utils :refer [inspect sym-map]]))

(def ByteBuffer js/ByteBuffer)

(defrecord MutableOutputStream [buf])

(defn write-long-varint-zz [this l]
  (.writeVarint64ZigZag (:buf this) l)
  nil)

(defn get-size [this]
  (.-offset (:buf this)))

(defn write-byte [this b]
  (.writeInt8 (:buf this) b)
  nil)

(defn write-bytes [this bs num-bytes]
  (dotimes [n num-bytes]
    (write-byte this (aget bs n))))

(defn write-bytes-w-len-prefix [this bs]
  (let [num-bytes (count bs)]
    (write-long-varint-zz this num-bytes)
    (write-bytes this bs num-bytes)))

(defn write-utf8-string [this s]
  (let [num-bytes (.calculateUTF8Bytes ByteBuffer s)]
    (write-long-varint-zz this num-bytes)
    (.writeUTF8String (:buf this) s)
    nil))

(defn write-float [this f]
  (.writeFloat32 (:buf this) f)
  nil)

(defn write-double [this d]
  (.writeFloat64 (:buf this) d)
  nil)

(defn to-byte-array [this]
  (.flip (:buf this))
  (-> (.toArrayBuffer (:buf this))
      (js/Int8Array.)))

(defn to-b64-string [this]
  (.flip (:buf this))
  (.toBase64 (:buf this)))

(defn to-utf8-string [this]
  (.flip (:buf this))
  (.toUTF8 (:buf this)))


(defrecord MutableInputStream [buf])

(defn read-long-varint-zz [this]
  (.readVarint64ZigZag (:buf this)))

(defn get-available-count [this]
  (.remaining (:buf this)))

(defn read-byte [this]
  (.readInt8 (:buf this)))

(defn read-bytes [this num-bytes]
  (let [ab (js/ArrayBuffer. num-bytes)
        view (js/Int8Array. ab)]
    (dotimes [n num-bytes]
      (->> (read-byte this)
           (aset view n)))
    view))

(defn read-len-prefixed-bytes [this]
  (let [num-bytes (read-long-varint-zz this)]
    (read-bytes this num-bytes)))

(defn read-utf8-string [this]
  (let [num-bytes (-> (read-long-varint-zz this)
                      (.toInt))]
    (.readUTF8String (:buf this) num-bytes (.-METRICS_BYTES ByteBuffer))))

(defn read-float [this]
  (.readFloat32 (:buf this)))

(defn read-double [this]
  (.readFloat64 (:buf this)))

(defn byte-array->mutable-input-stream [bs]
  (->MutableInputStream (.wrap ByteBuffer (.-buffer bs) "utf8" true)))

(defn b64-string->mutable-input-stream [s]
  (->MutableInputStream (ByteBuffer.fromBase64 s true)))

(defn make-mutable-output-stream []
  (->MutableOutputStream (ByteBuffer. 16 true)))
