(ns farbetter.roe.serdes
  (:refer-clojure :exclude [long])
  (:require
   [clojure.set :refer [difference]]
   [clojure.string :as str]
   [farbetter.roe.io-streams :as ios]
   [farbetter.roe.fingerprint :as f]
   [farbetter.roe.mutable-streams :as ms :refer [InputStream OutputStream]]
   [farbetter.roe.schemas :refer
    [AvroArray AvroArraySchema AvroBoolean AvroData AvroEnum AvroEnumSchema
     AvroFieldSchema AvroFixedOrBytes AvroFixedSchema
     AvroLong AvroMapSchema AvroName AvroNull AvroNum AvroRecordOrMap
     AvroRecordSchema AvroSchema AvroString AvroUnionSchema
     avro-built-in-types avro-named-types
     avro-name-pattern-str avro-primitive-types
     avro-recursive-types valid-avro-name?]]
   [farbetter.roe.utils :as ru :refer
    [#?@(:cljs [float?]) get-avro-type]]
   [farbetter.utils :as u :refer
    [byte-array? long long? throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])
     :clj
     (:import [clojure.lang ExceptionInfo])))

(declare add-schema decode-long decode-edn
         encode-long encode-edn)

(def max-bytes-per-avro-obj 1e6)

;;;;;;;;;;;;;;;;;;;; Local Schemas ;;;;;;;;;;

(def SchemaNameToSchema {AvroName AvroSchema})
(def EnumName AvroName)
(def EnumSymbol AvroName)
(def EnumSymbolPos s/Int)
(def RecordName AvroName)
(def FieldName AvroName)
(def FieldMap {FieldName AvroFieldSchema})
(def EnumSymbolToPosKey (s/pair EnumName "enum-name"
                                EnumSymbol "enum-symbol"))
(def EnumPosToSymbolKey (s/pair EnumName "enum-name"
                                EnumSymbolPos "enum-symbol-pos"))
(def Context
  {(s/optional-key :name->schema) SchemaNameToSchema
   (s/optional-key :enum-symbol->pos) {EnumSymbolToPosKey EnumSymbolPos}
   (s/optional-key :enum-pos->symbol) {EnumPosToSymbolKey EnumSymbol}
   (s/optional-key :record->fields) {RecordName #{FieldName}}
   (s/optional-key :record->field-map) {RecordName FieldMap}
   (s/optional-key :record->alias-map) {RecordName FieldMap}})

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;

(defn expand-schema [schema context]
  (let [avro-type (get-avro-type schema)]
    (if (contains? avro-built-in-types avro-type)
      schema
      (get-in context [:name->schema schema]))))


(s/defn encode-string :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (string? data)
    (throw-far-error "Data is not a string"
                     :schema-violation :invalid-string
                     (sym-map data)))
  (ms/write-utf8-string os data))

(s/defn decode-string :- AvroData
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (let [read-fn (case (get-avro-type reader-schema)
                  :bytes ms/read-len-prefixed-bytes
                  :string ms/read-utf8-string
                  (throw-far-error
                   (str "Schema mismatch. Writer schema is :string and reader "
                        "schema is `" reader-schema "`")
                   :schema-mismatch :reader-schema-not-compatible-with-string
                   {:writer-schema :string :reader-schema reader-schema}))]
    (read-fn is)))

(s/defn encode-bytes :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (byte-array? data)
    (throw-far-error "Data is not a byte array"
                     :schema-violation :invalid-byte-array
                     (sym-map data)))
  (ms/write-bytes-w-len-prefix os data))

(s/defn decode-bytes :- AvroData
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (if-not (= :bytes (get-avro-type reader-schema))
    (throw-far-error
     (str "Schema mismatch. Writer schema is :bytes and reader "
          "schema is `" reader-schema "`")
     :schema-mismatch :reader-schema-not-compatible-with-bytes
     {:writer-schema :bytes :reader-schema reader-schema}))
  (ms/read-len-prefixed-bytes is))

(s/defn encode-double :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (#?(:clj float? :cljs number?) data)
    (throw-far-error "Data is not a floating point number"
                     :schema-violation :invalid-floating-point-number
                     (sym-map data)))
  (ms/write-double os data))

(s/defn decode-double :- AvroNum
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (when-not (= :double (get-avro-type reader-schema))
    (throw-far-error
     (str "Schema mismatch. Writer schema is :double and reader "
          "schema is `" reader-schema "`")
     :schema-mismatch :reader-schema-not-compatible-with-double
     {:writer-schema :double :reader-schema reader-schema}))
  (ms/read-double is))

(s/defn encode-float :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (#?(:clj float? :cljs number?) data)
    (throw-far-error "Data is not a floating point number"
                     :schema-violation :invalid-floating-point-number
                     (sym-map data)))
  (when (or (> data 3.4028234E38)
            (< data -3.4028234E38))
    (throw-far-error "Value out of range for float"
                     :schema-violation :value-out-of-range-for-float
                     (sym-map data)))
  (ms/write-float os data))

(s/defn decode-float :- AvroNum
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :float identity
                  :double double
                  (throw-far-error
                   (str "Schema mismatch. Writer schema is :float and reader "
                        "schema is `" reader-schema "`")
                   :schema-mismatch :reader-schema-not-compatible-with-float
                   {:writer-schema :float :reader-schema reader-schema}))
        data (ms/read-float is)]
    (cast-fn data)))

(s/defn encode-long :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (or (long? data)
                (integer? data))
    (throw-far-error "Data is not a long"
                     :schema-violation :invalid-long
                     (sym-map data)))
  (when (or (> data 9223372036854775807)
            (< data -9223372036854775808))
    (throw-far-error "Value out of range for long"
                     :schema-violation :value-out-of-range-for-long
                     (sym-map data)))
  (ms/write-long-varint-zz os data))

(s/defn decode-long :- AvroNum
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :long long
                  :float float
                  :double double
                  (throw-far-error
                   (str "Schema mismatch. Writer schema is :long and reader "
                        "schema is `" reader-schema "`")
                   :schema-mismatch :reader-schema-not-compatible-with-long
                   {:writer-schema :long :reader-schema reader-schema}))
        num (ms/read-long-varint-zz is)]
    (cast-fn num)))

(s/defn encode-int :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (integer? data)
    (throw-far-error "Data is not an integer"
                     :schema-violation :invalid-integer
                     (sym-map data)))
  (when (or (> data 2147483647)
            (< data -2147483648))
    (throw-far-error "Value out of range for integer"
                     :schema-violation :value-out-of-range-for-integer
                     (sym-map data)))
  (ms/write-long-varint-zz os data))

(s/defn decode-int :- AvroNum
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :int #?(:clj int :cljs #(.toInt %))
                  :long identity
                  :float #?(:clj float :cljs #(.toNumber %))
                  :double #?(:clj double :cljs #(.toNumber %))
                  (throw-far-error
                   (str "Schema mismatch. Writer schema is :int and reader "
                        "schema is `" reader-schema "`")
                   :schema-mismatch :reader-schema-not-compatible-with-int
                   {:writer-schema :int :reader-schema reader-schema}))
        num (ms/read-long-varint-zz is)]
    (cast-fn num)))

(s/defn encode-boolean :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (ms/write-byte os (case data
                      false 0
                      true 1
                      (throw-far-error "Data is not boolean"
                                       :schema-violation :invalid-boolean
                                       (sym-map data)))))

(s/defn decode-boolean :- AvroBoolean
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (when-not (= :boolean (get-avro-type reader-schema))
    (throw-far-error
     (str "Schema mismatch. Writer schema is :boolean and reader schema is `"
          reader-schema "`")
     :schema-mismatch :reader-schema-not-boolean
     {:writer-schema :boolean :reader-schema reader-schema}))
  (let [data (ms/read-byte is)]
    (case (int data)
      0 false
      1 true
      (throw-far-error "Invalid data received"
                       :invalid-data :invalid-boolean
                       (sym-map data)))))

(s/defn encode-null :- (s/eq nil)
  "Nulls are encoded as zero bytes."
  [os :- (s/protocol OutputStream)
   data :- AvroData]
  (when-not (nil? data)
    (throw-far-error "Data is not nil (schema is :null)"
                     :schema-violation :data-not-nil
                     (sym-map data))))

(s/defn decode-null :- AvroNull
  "Nulls are encoded as zero bytes."
  [is :- (s/protocol InputStream)
   reader-schema :- AvroSchema]
  (when-not (= :null (get-avro-type reader-schema))
    (throw-far-error
     (str "Schema mismatch. Writer schema is :null and reader schema is `"
          reader-schema "`")
     :schema-mismatch :reader-schema-not-null
     {:writer-schema :null :reader-schema reader-schema}))
  nil)

(defn- trial-encode [os item union-schema union-pos context]
  (let [item-schema (union-schema union-pos)]
    (try
      (ms/write-long-varint-zz os union-pos)
      (encode-edn os item item-schema context)
      true
      (catch ExceptionInfo e
        (if (= :schema-violation (:type (ex-data e)))
          :failed-to-encode
          (throw e))))))

(s/defn encode-union :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   union-schema :- AvroUnionSchema
   context :- Context]
  (let [union-count (count union-schema)]
    (loop [union-pos 0]
      (when (>= union-pos union-count)
        (throw-far-error "Data does not match union schema"
                         :schema-violation :invalid-union
                         (sym-map data union-schema)))
      (let [trial-os (ios/make-mutable-output-stream)
            result (trial-encode trial-os data union-schema union-pos context)]
        (if (= :failed-to-encode result)
          (recur (inc union-pos))
          (let [size (ms/get-size trial-os)
                bytes (ms/to-byte-array trial-os)]
            (ms/write-bytes os bytes size)))))))

(defn- trial-decode [is writer-schema writer-context
                     reader-schema reader-context]
  (try
    (decode-edn is writer-schema writer-context reader-schema reader-context)
    (catch ExceptionInfo e
      (if (= :schema-mismatch (:type (ex-data e)))
        :failed-to-decode
        (throw e)))))

(s/defn decode-w-union-reader :- AvroData
  [is :- (s/protocol InputStream)
   writer-schema :- AvroSchema
   writer-context :- Context
   reader-schema :- AvroUnionSchema
   reader-context :- Context]
  (ms/mark is max-bytes-per-avro-obj)
  (let [union-count (count reader-schema)]
    (loop [union-pos 0]
      (when (>= union-pos union-count)
        (throw-far-error "No reader schemas in union match written data"
                         :schema-mismatch :no-reader-schemas-match
                         (sym-map reader-schema writer-schema)))
      (let [data (trial-decode is writer-schema writer-context
                               (reader-schema union-pos) reader-context)]
        (if-not (= :failed-to-decode data)
          data
          (do
            (ms/reset is)
            (recur (inc union-pos))))))))

(s/defn decode-union :- AvroData
  [is :- (s/protocol InputStream)
   writer-schema :- AvroUnionSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (let [union-pos (decode-long is :long)
        writer-item-schema (writer-schema union-pos)]
    (decode-edn is writer-item-schema writer-context
                reader-schema reader-context)))

;; TODO: Encode as multiple blocks? (optional)
(s/defn encode-map :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroMapSchema
   context :- Context]
  (when-not (map? data)
    (throw-far-error "Data is not a map"
                     :schema-violation :invalid-map
                     (sym-map data)))
  (let [value-schema (:values schema)
        encode-kvp (fn [acc k v]
                     (when-not (string? k)
                       (throw-far-error "Map keys must be strings"
                                        :schema-violation :invalid-map-key
                                        {:map data :bad-key k}))
                     (encode-string os k)
                     (encode-edn os v value-schema context))
        num-kvps (count data)]
    (when (pos? num-kvps)
      (encode-long os num-kvps)
      (reduce-kv encode-kvp nil data))
    (encode-long os 0)))

(defn decode-map-block [is m block-count writer-value-schema writer-context
                        reader-value-schema reader-context]
  (let [decode-kvp (fn [m n]
                     (let [k (decode-string is :string)
                           v (decode-edn
                              is writer-value-schema writer-context
                              reader-value-schema reader-context)]
                       (assoc m k v)))]
    (reduce decode-kvp m (range block-count))))

(s/defn decode-map :- AvroRecordOrMap
  [is :- (s/protocol InputStream)
   writer-schema :- AvroMapSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (let [writer-value-schema (:values writer-schema)
        reader-value-schema (:values reader-schema)
        _ (when-not (= (get-avro-type writer-value-schema)
                       (get-avro-type reader-value-schema))
            (throw-far-error
             (str "Schema mismatch. Writer value schema is `"
                  writer-value-schema
                  "` and reader value schema is `" reader-value-schema)
             :schema-mismatch :map-value-schema-mismatch
             (sym-map writer-schema reader-schema)))
        decode-block (fn [m block-count]
                       (decode-map-block
                        is m block-count writer-value-schema writer-context
                        reader-value-schema reader-context))
        decode-sized-block (fn [m block-count]
                             (let [num-bytes (decode-long is :long)]
                               (decode-block m block-count)))]
    (loop [m {}]
      (let [block-count (decode-long is :long)]
        (cond
          (pos? block-count) (recur (decode-block m block-count))
          (neg? block-count) (recur (decode-sized-block m
                                                        (u/abs block-count)))
          :else m)))))

;; TODO: Encode as multiple blocks? (optional)
(s/defn encode-array :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   array :- AvroData
   schema :- AvroArraySchema
   context :- Context]
  (when-not (sequential? array)
    (throw-far-error "Data is not an array"
                     :schema-violation :invalid-array
                     {:data array}))
  (let [item-schema (:items schema)
        num-items (count array)
        encode-item (fn [acc item]
                      (encode-edn os item item-schema context))]
    (when (pos? num-items)
      (encode-long os num-items)
      (reduce encode-item nil array))
    (encode-long os 0)))

(defn decode-array-block [is array block-count writer-item-schema writer-context
                          reader-item-schema reader-context]
  (let [decode-item (fn [array n]
                      (let [edn (decode-edn
                                 is writer-item-schema writer-context
                                 reader-item-schema reader-context)]
                        (conj array edn)))]
    (reduce decode-item array (range block-count))))

(s/defn decode-array :- AvroArray
  [is :- (s/protocol InputStream)
   writer-schema :- AvroArraySchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (let [writer-item-schema (:items writer-schema)
        reader-item-schema (:items reader-schema)
        _ (when-not (= (get-avro-type writer-item-schema)
                       (get-avro-type reader-item-schema))
            (throw-far-error
             (str "Schema mismatch. Writer item schema is `" writer-item-schema
                  "` and reader item schema is `" reader-item-schema)
             :schema-mismatch :array-item-schema-mismatch
             (sym-map writer-schema reader-schema)))
        decode-block (fn [array block-count]
                       (decode-array-block
                        is array block-count writer-item-schema writer-context
                        reader-item-schema reader-context))
        decode-sized-block (fn [array block-count]
                             (let [num-bytes (decode-long is :long)]
                               (decode-block array block-count)))]
    (loop [array []]
      (let [block-count (decode-long is :long)]
        (cond
          (pos? block-count) (recur (decode-block array block-count))
          (neg? block-count) (recur (decode-sized-block array
                                                        (u/abs block-count)))
          :else array)))))

(defn check-named-schemas [writer-schema reader-schema]
  (let [writer-type (get-avro-type writer-schema)
        reader-type (get-avro-type reader-schema)
        writer-name (:name writer-schema)
        reader-name (:name reader-schema)]
    (when-not (= writer-type reader-type)
      (throw-far-error
       (str "Schema mismatch. Writer schema is type " writer-type
            " and reader schema is type " reader-type)
       :schema-mismatch :different-schema-types
       (sym-map writer-schema reader-schema)))
    (when-not (or (= writer-name reader-name)
                  (some #{writer-name} (:aliases reader-schema)))
      (throw-far-error
       (str "Schema mismatch. Reader and writer schemas do not have the "
            "same name. Reader name: `" (:name reader-schema)
            "`. Writer name: `" name "`")
       :schema-mismatch :different-schema-names
       (sym-map writer-schema reader-schema)))))

(s/defn encode-fixed :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroFixedSchema
   context :- Context]
  (when-not (byte-array? data)
    (throw-far-error "Data is not a fixed byte array"
                     :schema-violation :invalid-fixed
                     (sym-map data)))
  (let [{:keys [size]} schema
        _ (when-not (= (count data) size)
            (throw-far-error "Size of fixed does not match schema"
                             :schema-violation :illegal-size-of-fixed
                             {:schema-size size :data-size (count data)}))]
    (ms/write-bytes os data size)))

(s/defn decode-fixed :- AvroFixedOrBytes
  [is :- (s/protocol InputStream)
   writer-schema :- AvroFixedSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (check-named-schemas writer-schema reader-schema)
  (let [writer-size (:size writer-schema)
        reader-size (:size reader-schema)]
    (when-not (= writer-size reader-size)
      (throw-far-error
       (str "Schema mismatch. Writer size is " writer-size
            " and reader size is " reader-size)
       :schema-mismatch :fixed-size-mismatch
       (sym-map writer-schema reader-schema)))
    (ms/read-bytes is (int writer-size))))

(s/defn encode-enum :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroEnumSchema
   context :- Context]
  (let [{:keys [name]} schema
        pos (get-in context [:enum-symbol->pos [name data]])]
    (when-not pos
      (throw-far-error (str "`" data "` is not a valid symbol keyword "
                            "for this enum")
                       :schema-violation :invalid-enum-symbol
                       (sym-map symbol schema)))
    (encode-int os pos)))

(s/defn decode-enum :- AvroEnum
  [is :- (s/protocol InputStream)
   writer-schema :- AvroEnumSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (check-named-schemas writer-schema reader-schema)
  (let [reader-symbols (:symbols reader-schema)
        pos  (decode-int is :int)
        written-symbol (get-in writer-context [:enum-pos->symbol
                                               [(:name writer-schema) pos]])]
    (when-not (contains? reader-symbols written-symbol)
      (throw-far-error "Written symbol does not appear in reader's symbol set"
                       :schema-mismatch :unreadable-symbol
                       (sym-map reader-symbols written-symbol writer-schema
                                reader-schema)))
    written-symbol))

(s/defn encode-record :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   record :- AvroData
   schema :- AvroRecordSchema
   context :- Context]
  (when-not (map? record)
    (throw-far-error "Data is not a record"
                     :schema-violation :invalid-record
                     {:data record}))
  (let [{:keys [fields name]} schema
        schema-fields (get-in context [:record->fields name])
        record-fields (set (keys record))
        extra-fields (difference record-fields schema-fields)
        _ (when (seq extra-fields)
            (throw-far-error "Extra record fields found"
                             :schema-violation :extra-record-fields
                             (sym-map extra-fields schema record)))
        encode-field (fn [acc field]
                       (let [{:keys [type name default]} field
                             field-value (if (contains? record name)
                                           (record name)
                                           default)]
                         (encode-edn os field-value type context)))]
    (reduce encode-field nil fields)))

(defn resolve-record [writer-schema writer-context reader-schema reader-context
                      written-rec]
  (let [reader-name (:name reader-schema)
        written-field-names (set (keys written-rec))
        reader-field-names (get-in reader-context [:record->fields reader-name])
        added-fields #(difference reader-field-names written-field-names)
        add-field (fn [acc field-name]
                    (let [reader-field (get-in reader-context
                                               [:record->field-map reader-name
                                                field-name])]
                      (assoc acc field-name (:default reader-field))))
        add-fields (fn [m field-names]
                     (reduce add-field m field-names))]
    (if (= written-field-names reader-field-names)
      written-rec
      (-> (select-keys written-rec reader-field-names)
          (add-fields (added-fields))))))

(s/defn decode-record :- AvroRecordOrMap
  [is :- (s/protocol InputStream)
   writer-schema :- AvroRecordSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (check-named-schemas writer-schema reader-schema)
  (let [record-name (:name writer-schema)
        ;; Get the reader's field, incl. if it is an alias
        get-reader-field (fn [name]
                           (if-let [field (get-in reader-context
                                                  [:record->field-map
                                                   record-name name])]
                             field
                             (get-in reader-context
                                     [:record->alias-map record-name name])))
        add-field (fn [rec writer-field]
                    (let [{:keys [name type]} writer-field
                          ;; If the reader doesn't have this field,
                          ;; read using the writer's schema
                          reader-field (or (get-reader-field name)
                                           writer-field)
                          field-name (:name reader-field)
                          value  (decode-edn is type writer-context
                                             (:type reader-field)
                                             reader-context)]
                      (assoc rec field-name value)))
        written-rec  (reduce add-field {} (:fields writer-schema))]
    (if (= writer-schema reader-schema)
      written-rec
      (resolve-record writer-schema writer-context reader-schema
                      reader-context written-rec))))

(s/defn encode-edn :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   edn :- AvroData
   schema :- AvroSchema
   context :- Context]
  (let [schema (expand-schema schema context)
        avro-type (get-avro-type schema)
        enc-fn (case avro-type
                 :record encode-record
                 :enum encode-enum
                 :array encode-array
                 :map encode-map
                 :fixed encode-fixed
                 :union encode-union
                 :null encode-null
                 :boolean encode-boolean
                 :int encode-int
                 :long encode-long
                 :float encode-float
                 :double encode-double
                 :bytes encode-bytes
                 :string encode-string)]
    (if (contains? avro-primitive-types avro-type)
      (enc-fn os edn)
      (enc-fn os edn schema context)))
  (let [num-bytes (ms/get-size os)]
    (when (> num-bytes max-bytes-per-avro-obj)
      (throw-far-error (str "Serialized data exceeds maximum size limit of "
                            max-bytes-per-avro-obj " bytes")
                       :runtime-error :serialized-data-too-large
                       (sym-map max-bytes-per-avro-obj num-bytes)))))

(s/defn decode-edn :- AvroData
  [is :- (s/protocol InputStream)
   writer-schema :- AvroSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (let [writer-schema (expand-schema writer-schema writer-context)
        reader-schema (expand-schema reader-schema reader-context)
        writer-avro-type (get-avro-type writer-schema)
        reader-avro-type (get-avro-type reader-schema)
        dec-fn (case writer-avro-type
                 :record decode-record
                 :enum decode-enum
                 :array decode-array
                 :map decode-map
                 :fixed decode-fixed
                 :union decode-union
                 :null decode-null
                 :boolean decode-boolean
                 :int decode-int
                 :long decode-long
                 :float decode-float
                 :double decode-double
                 :bytes decode-bytes
                 :string decode-string)]
    (cond
      (= :union reader-avro-type)
      (decode-w-union-reader is writer-schema writer-context
                             reader-schema reader-context)

      (contains? avro-primitive-types writer-avro-type)
      (dec-fn is reader-schema)

      :else
      (dec-fn is writer-schema writer-context reader-schema reader-context))))

(defn get-child-schemas [schema avro-type]
  (case avro-type
    :record (map :type (:fields schema))
    :array [(:items schema)]
    :map [(:values schema)]
    :union schema
    []))

(defn add-child-schemas [context schema avro-type]
  (let [child-schemas (get-child-schemas schema avro-type)
        add-child (fn [context child-schema]
                    (when (and (= :union avro-type)
                               (= :union (get-avro-type child-schema)))
                      (throw-far-error
                       "Unions may not immediately contain other unions"
                       :illegal-schema :illegal-union-nesting
                       (sym-map schema child-schema context)))
                    (add-schema context child-schema))]
    (reduce add-child context child-schemas)))

(defn add-name [context [name schema]]
  (let [{:keys [name->schema]} context]
    (when-not (valid-avro-name? name)
      (throw-far-error (str "`" name "` is not a valid name keyword. "
                            "Names must be keywords whose names match this "
                            "regex: " avro-name-pattern-str " to be valid")
                       :illegal-schema :illegal-avro-name
                       (sym-map name schema)))
    (when (and (contains? name->schema name)
               (not= schema (name->schema name)))
      (throw-far-error (str "Different schemas have the same name: `" name)
                       :illegal-schema :same-name-for-different-schemas
                       (sym-map name schema context)))
    (assoc-in context [:name->schema name] schema)))


(defn add-enum-info [context name schema]
  (let [{:keys [symbols]} schema
        f (fn [context [pos symbol]]
            (when-not (valid-avro-name? symbol)
              (throw-far-error
               (str "Illegal enum symbol `" symbol "` in schema. Symbols must "
                    "be keywords whose names match this regex: "
                    avro-name-pattern-str " to be valid")
               :illegal-schema :illegal-enum-symbol
               (sym-map schema symbol)))
            (-> context
                (assoc-in [:enum-symbol->pos [name symbol]] pos)
                (assoc-in [:enum-pos->symbol [name pos]] symbol)))]
    (when-not (seq symbols)
      (throw-far-error "Illegal enum schema. Symbol set is empty"
                       :illegal-schema :empty-enum-symbol-set
                       (sym-map schema)))
    (reduce f context (map-indexed vector (sort symbols)))))

(defn check-record-info [record-name schema field-names field]
  (let [field-name (:name field)]
    (when (contains? field-names field-name)
      (throw-far-error
       (str "Duplicate field name `" field-name
            "` in record schema")
       :illegal-schema :duplicate-field-name
       (sym-map record-name schema field-name)))
    (when-not (valid-avro-name? field-name)
      (throw-far-error
       (str "Illegal field name `" field-name
            "` in record schema. Field names must be "
            "keywords whose names match this regex: "
            avro-name-pattern-str " to be valid")
       :illegal-schema :illegal-record-field-name
       (sym-map schema record-name field-name)))
    (when-not (contains? field :default)
      (throw-far-error
       (str "Field `" field-name "` does not have a default value")
       :illegal-schema :missing-default
       (sym-map schema field-name)))))

(defn update-aliases [context record-name aliases field]
  (let [update-alias (fn [context alias-name]
                       (assoc-in context
                                 [:record->alias-map record-name alias-name]
                                 field))]
    (reduce update-alias context aliases)))

(defn add-record-info [context record-name schema]
  (let [add-field (fn [context field]
                    (let [{:keys [name aliases]} field
                          field-names (get-in context
                                              [:record->fields
                                               record-name])]
                      (check-record-info record-name schema field-names field)
                      (-> (update-in context [:record->fields record-name]
                                     #(conj (or % #{}) name))
                          (assoc-in [:record->field-map record-name name]
                                    field)
                          (update-aliases record-name aliases field))))]
    (reduce add-field context (:fields schema))))

(defn add-enum-and-record-info [context]
  (let [{:keys [name->schema]} context
        f (fn [context name schema]
            (case (get-avro-type schema)
              :enum (add-enum-info context name schema)
              :record (add-record-info context name schema)
              context))]
    (reduce-kv f context name->schema)))

(defn add-named-schema [context schema]
  (let [{:keys [name aliases]} schema
        all-names (conj aliases name)
        name-schema-pairs (map #(vector % schema) all-names)]
    (reduce add-name context name-schema-pairs)))

(defn add-schema [context schema]
  (let [avro-type (get-avro-type schema)
        named? (contains? avro-named-types avro-type)
        recursive? (contains? avro-recursive-types avro-type)]
    (when (and (= :union avro-type)
               (not= (count schema) (count (set schema))))
      (throw-far-error "Illegal union schema. Members are not unique"
                       :illegal-schema :non-unique-union-members
                       (sym-map schema)))
    (cond-> context
      named? (add-named-schema schema)
      recursive? (add-child-schemas schema avro-type))))

(s/defn make-context-impl :- Context
  [top-schema :- AvroSchema]
  (-> (add-schema {} top-schema)
      (add-enum-and-record-info)))

(def make-context (memoize make-context-impl))

(s/defn make-default-record :- AvroData
  [schema :- AvroRecordSchema]
  ;; Check schema validity
  (make-context schema)
  (when-not (= :record (:type schema))
    (throw-far-error "make-default-record can only be called on record schemas"
                     :illegal-argument :not-a-record-schema
                     (sym-map schema)))
  (let [add-field (fn [acc {:keys [type name default]}]
                    (assoc acc name (if (= :record (get-avro-type type))
                                      (make-default-record type)
                                      default)))]
    (reduce add-field {} (:fields schema))))

(s/defn check-schema :- nil
  [schema :- AvroSchema]
  ;; make-context does schema checking, throwing :illegal-schema as needed
  (make-context schema))

(s/defn valid-schema? :- s/Bool
  [schema :- AvroSchema]
  (try
    (check-schema schema)
    true
    (catch ExceptionInfo e
      (if (= :illegal-schema (:type (ex-data e)))
        false
        (throw e)))))

(defn edn->output-stream [schema edn]
  (let [context (make-context schema)
        os (ios/make-mutable-output-stream)]
    (encode-edn os edn schema context)
    os))

(defn input-stream->edn [input-stream writer-schema reader-schema]
  (let [writer-context (make-context writer-schema)
        reader-context (make-context reader-schema)
        edn (decode-edn input-stream writer-schema writer-context
                        reader-schema reader-context)
        remaining-count (ms/get-available-count input-stream)]
    (when-not (zero? remaining-count)
      (throw-far-error "Extra items in stream"
                       :illegal-argument :extra-items-in-stream
                       (sym-map edn remaining-count
                                reader-schema writer-schema)))
    edn))

(s/defn edn->avro-byte-array :- AvroFixedOrBytes
  [schema :- s/Any ;; AvroSchema
   edn :- AvroData]
  (-> (edn->output-stream schema edn)
      (ms/to-byte-array)))

(s/defn edn->avro-b64-string :- s/Str
  [schema :- AvroSchema
   edn :- AvroData]
  (-> (edn->output-stream schema edn)
      (ms/to-b64-string)))

(s/defn avro-byte-array->edn :- AvroData
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-byte-array :- AvroFixedOrBytes]
  (-> (ios/byte-array->mutable-input-stream avro-byte-array)
      (input-stream->edn writer-schema reader-schema)))

(s/defn avro-b64-string->edn :- AvroData
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-b64-string :- s/Str]
  (-> (ios/b64-string->mutable-input-stream avro-b64-string)
      (input-stream->edn writer-schema reader-schema)))
