(ns farbetter.roe
  (:require
   [farbetter.roe.bencode :as bc]
   [farbetter.roe.fingerprint :as f]
   [farbetter.roe.json :as json]
   [farbetter.roe.schemas :refer [AvroData AvroFixedOrBytes AvroSchema]]
   [farbetter.roe.serdes :as serdes]
   [farbetter.utils :as u :refer [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(def LongType (s/pred u/long?))

(s/defn edn->avro-byte-array :- AvroFixedOrBytes
  "Encodes an EDN data structure into an Avro byte array."
  [schema :- AvroSchema
   edn :- AvroData]
  (serdes/edn->avro-byte-array schema edn))

(s/defn avro-byte-array->edn :- AvroData
  "Decodes an Avro byte array into an EDN data structure."
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-byte-array :- AvroFixedOrBytes]
  (serdes/avro-byte-array->edn writer-schema reader-schema avro-byte-array))

(s/defn edn->avro-b64-string :- s/Str
  "Encodes a Clojure data structure into an Avro base-64 string."
  [schema :- AvroSchema
   edn :- AvroData]
  (serdes/edn->avro-b64-string schema edn))

(s/defn avro-b64-string->edn :- AvroData
  "Decodes an Avro base-64 string into a Clojure data structure."
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-str :- s/Str]
  (serdes/avro-b64-string->edn writer-schema reader-schema avro-str))

(s/defn check-schema :- nil
  "Checks schema for validity. Throws :illegal-schema exceptions if invalid."
  [schema :- AvroSchema]
  (serdes/check-schema schema))

(s/defn valid-schema? :- s/Bool
  "Checks schema for validity. Returns true or false."
  [schema :- AvroSchema]
  (serdes/valid-schema? schema))

(s/defn edn-schema->fingerprint :- LongType
  "Returns a 64-bit integer representing the fingerprint of the schema."
  [schema :- AvroSchema]
  (check-schema schema)
  (let [bencoded-canonical-form (bc/edn-schema->bencode schema true)]
    (f/fingerprint64 bencoded-canonical-form)))

(s/defn json-schema->edn-schema :- AvroSchema
  "Translates a JSON Avro schema into an EDN Avro schema."
  [s :- s/Str]
  (json/json->edn-schema s))

(s/defn edn-schema->json-schema :- s/Str
  "Translates an EDN Avro schema into a JSON Avro schema."
  [schema :- AvroSchema]
  (json/edn-schema->json schema))

(s/defn make-default-record :- AvroData
  [schema :- AvroSchema]
  "Makes a default record for the given record schema."
  (serdes/make-default-record schema))
