(ns farbetter.roe.transform
  (:require
   [clojure.string :as str]
   [farbetter.roe.io-streams :as ios]
   [farbetter.roe.mutable-streams :as ms]
   [farbetter.roe.schemas :as sch]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map] :cljs [float?])]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(defn byte-array->b64 [bytes]
  (let [mos (ios/make-mutable-output-stream)
        num-bytes (count bytes)]
    (ms/write-bytes mos bytes num-bytes)
    (ms/to-b64-string mos)))

(defn b64->byte-array [b64]
  (let [mis (ios/b64-string->mutable-input-stream b64)
        len (ms/get-available-count mis)]
    (ms/read-bytes mis len)))

(defn transform-schema
  "Transform a clojure/script representation of the schema.
  If encode-or-decode is set to :encode, the input schema is assumed to be
  an edn-format schema, with keyword names which use  - instead of _.
  If encode-or-decode is set to :decode, the input schema is assumed to be
  an avro-format schema, with string names which use _ instead of -.
  The output schema will be the opposite type of the input schema.
  If canonicalize? is true, extra fields (doc and aliases) will be stripped."
  [s encode-or-decode canonicalize?]
  (when-not (#{:encode :decode} encode-or-decode)
    (throw-far-error "Bad encode-or-decode value"
                     :illegal-argument :bad-encode-or-decode-value
                     (sym-map encode-or-decode)))
  (letfn [(get [m k]
            (case encode-or-decode
              :encode (m k)
              :decode (m (name k))))
          (update-key [m k f]
            (let [[source-key dest-key] (case encode-or-decode
                                          :encode [k (name k)]
                                          :decode [(name k) k])
                  source-val (m source-key)
                  dest-val (f source-val)]
              (-> (assoc m dest-key dest-val)
                  (dissoc m source-key))))
          (remove-key [m k]
            (let [source-key (case encode-or-decode
                               :encode k
                               :decode (name k))]
              (dissoc m source-key)))
          (xf-name [n]
            (case encode-or-decode
              :encode (-> (name n)
                          (str/replace \- \_))
              :decode (-> (str/replace n \_ \-)
                          (keyword))))
          (xf-schema [s]
            (cond
              (sequential? s) (xf-vec s)
              (map? s) (xf-typed s)
              :else (xf-name s)))
          (xf-vec [v]
            (mapv xf-schema v))
          (xf-prim-map [m]
            (m "type"))
          (xf-typed [m]
            (let [type (name (get m :type))
                  xf-fn (if (sch/avro-primitive-type-names type)
                          xf-prim-map
                          (case type
                            "record" xf-record
                            "enum" xf-enum
                            "fixed" xf-fixed
                            "array" xf-array
                            "map" xf-map))]
              (-> (update-key m :type xf-name)
                  (xf-fn))))
          (xf-name-doc-aliases [m]
            (let [m (update-key m :name xf-name)]
              (if canonicalize?
                (-> m
                    (remove-key :aliases)
                    (remove-key :doc))
                (cond-> m
                  (get m :aliases) (update-key :aliases #(mapv xf-name %))
                  (get m :doc) (update-key :doc identity)))))
          (xf-map-value [m values-type]
            (let [xfmr (get-value-xfmr values-type)]
              (reduce-kv (fn [acc k v]
                           (assoc acc (xf-name k) (xfmr v)))
                         {} m)))
          (xf-array-value [arr element-type]
            (let [xfmr (get-value-xfmr element-type)]
              (mapv xfmr arr)))
          (xf-record-value [rec fields]
            (let [field-type-map (reduce (fn [acc field]
                                           (let [name (get field :name)
                                                 type (get field :type)]
                                             (assoc acc name type)))
                                         {} fields)]
              (reduce-kv (fn [acc k v]
                           (let [field-type (get field-type-map k)
                                 xfmr (get-value-xfmr field-type)]
                             (assoc acc (xf-name k) (xfmr v))))
                         {} rec)))
          (get-field-type-kw [field-type]
            (cond
              (map? field-type) (get-field-type-kw (get field-type :type))
              (keyword? field-type) field-type
              (string? field-type) (keyword field-type)
              :else (throw-far-error "Bad field type in record"
                                     :illegal-argument :bad-field-type
                                     (sym-map field-type))))
          (byte-array? [value]
            (case encode-or-decode
              :encode (u/byte-array? value)
              :decode (try
                        (u/byte-array? (b64->byte-array value))
                        (catch #?(:clj Exception :cljs :default) e
                          false))))
          (matches-type? [field-type value]
            (let [field-type-kw (get-field-type-kw field-type)]
              (case field-type-kw
                :null (case encode-or-decode
                        :encode (nil? value)
                        :decode (= "null" value))
                :boolean (case encode-or-decode
                           :encode (or (true? value)
                                       (false? value))
                           :decode (or (= "true" value)
                                       (= "false" value)))
                :int (case encode-or-decode
                       :encode (and (integer? value)
                                    (<= value 2147483647)
                                    (>= value -2147483648))
                       :decode (try
                                 (integer? (xf-int value))
                                 (catch #?(:clj Exception :cljs :default) e
                                   false)))
                :long (case encode-or-decode
                        :encode (or (integer? value)
                                    (u/long? value))
                        :decode (try
                                  (u/long? (xf-long value))
                                  (catch #?(:clj Exception :cljs :default) e
                                    false)))
                :float (case encode-or-decode
                         :encode (float? value)
                         :decode (try
                                   (float? (xf-float value))
                                   (catch #?(:clj Exception :cljs :default) e
                                     false)))
                :double (case encode-or-decode
                          :encode (float? value)
                          :decode (try
                                    (float? (xf-double value))
                                    (catch #?(:clj Exception :cljs :default) e
                                      false)))
                :bytes (byte-array? value)
                :string (string? value)
                :record (map? value)
                :enum (let [syms (get field-type :symbols)
                            syms-set (set syms)]
                        ;; In cljs, set lookup acts strangely
                        ;; with Long objects as the value, so
                        ;; we check the type of value first
                        (and (case encode-or-decode
                               :encode (keyword? value)
                               :decode (string? value))
                             (syms-set value)))
                :array (sequential? value)
                :map (map? value)
                :fixed (byte-array? value)
                (throw-far-error "Bad type in matches-type? fn"
                                 :illegal-argument :bad-type
                                 (sym-map field-type value)))))
          (xf-union-value [value union-type]
            (let [get-type (fn [result element-type]
                             (if (matches-type? element-type value)
                               (reduced element-type)
                               false))
                  element-type (reduce get-type false union-type)
                  _ (when (false? element-type)
                      (throw-far-error
                       "Value does not match any type in union."
                       :illegal-argument :bad-union-type
                       (sym-map value union-type)))
                  xfmr (get-value-xfmr element-type)]
              (xfmr value)))
          (xf-nil [v]
            (case encode-or-decode
              :encode "null"
              :decode nil))
          (xf-boolean [v]
            (case encode-or-decode
              :encode (case v
                        true "true"
                        false "false")
              :decode (case v
                        "true" true
                        "false" false)))
          (xf-int [v]
            (case encode-or-decode
              :encode #?(:clj (.toString ^Number v)
                         :cljs (.toString v))
              :decode #?(:clj (Integer/parseInt v)
                         :cljs (js/parseInt v))))
          (xf-long [v]
            (case encode-or-decode
              :encode (u/long->str (u/long v))
              :decode (u/str->long v)))
          (xf-float [v]
            (case encode-or-decode
              :encode (.toString ^Number v)
              :decode #?(:clj (Float/parseFloat v)
                         :cljs (js/parseFloat v))))
          (xf-double [v]
            (case encode-or-decode
              :encode (.toString ^Number v)
              :decode #?(:clj (Double/parseDouble v)
                         :cljs (js/parseFloat v))))
          (xf-bytes [v]
            (case encode-or-decode
              :encode (byte-array->b64 v)
              :decode (b64->byte-array v)))
          (get-value-xfmr [field-type]
            (if (vector? field-type)
              #(xf-union-value % field-type)
              (case (get-field-type-kw field-type)
                :null xf-nil
                :boolean xf-boolean
                :int xf-int
                :long xf-long
                :float xf-float
                :double xf-double
                :bytes xf-bytes
                :string identity
                :record #(xf-record-value % (get field-type :fields))
                :enum xf-name
                :array #(xf-array-value % (get field-type :items))
                :map #(xf-map-value % (get field-type :values))
                :fixed xf-bytes
                (throw-far-error "Bad type"
                                 :illegal-argument :bad-type-kw
                                 (sym-map field-type)))))
          (xf-field [f]
            (let [field-type (get f :type)
                  xf-default (get-value-xfmr field-type)]
              (-> (xf-name-doc-aliases f)
                  (update-key :type xf-schema)
                  (update-key :default xf-default)
                  (cond->
                      (get f :order) (update-key :order identity)))))
          (xf-fields [fs]
            (mapv xf-field fs))
          (xf-record [r]
            (-> (xf-name-doc-aliases r)
                (update-key :fields xf-fields)))
          (xf-enum [enum]
            (-> (xf-name-doc-aliases enum)
                (update-key :symbols #(map xf-name %))))
          (xf-array [a]
            (update-key a :items xf-schema))
          (xf-map [m]
            (update-key m :values xf-schema))
          (xf-fixed [f]
            (-> (xf-name-doc-aliases f)
                (update-key :size identity)))]
    (xf-schema s)))
