(ns farbetter.roe.serdes
  (:refer-clojure :exclude [long resolve])
  (:require
   #?(:cljs [cljs.pprint :refer [pprint]])
   [clojure.set :refer [difference]]
   [clojure.string :as str]
   [farbetter.roe.io-streams :as ios]
   [farbetter.roe.fingerprint :as f]
   [farbetter.roe.mutable-streams :as ms :refer [InputStream OutputStream]]
   [farbetter.roe.schemas :refer
    [AvroArray AvroArraySchema AvroBoolean AvroData AvroEnum AvroEnumSchema
     AvroFieldSchema AvroFixedOrBytes AvroFixedSchema
     AvroLong AvroMapSchema AvroName AvroNull AvroNum AvroRecordOrMap
     AvroRecordSchema AvroSchema AvroString AvroUnionSchema
     avro-built-in-types avro-named-types
     avro-name-pattern-str avro-primitive-types
     avro-recursive-types valid-avro-name?]]
   [farbetter.roe.utils :as ru :refer [get-avro-type union?]]
   [farbetter.stockroom :as stock]
   [farbetter.utils :as u :refer
    [byte-array? long long? throw-far-error
     #?@(:clj [inspect sym-map] :cljs [float?])]]
   #?(:clj [puget.printer :refer [cprint]])
   [schema.core :as s]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof tracef]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])
     :clj
     (:import [clojure.lang ExceptionInfo])))

(declare add-schema decode-edn encode-edn resolve)

(def max-bytes-per-avro-obj 1e9)
(def default-context-cache-size 1e4)

;;;;;;;;;;;;;;;;;;;; Local Schemas ;;;;;;;;;;

(def SchemaNameToSchema {AvroName AvroSchema})
(def EnumName AvroName)
(def EnumSymbol AvroName)
(def EnumSymbolPos s/Int)
(def RecordName AvroName)
(def FieldName AvroName)
(def FieldMap {FieldName AvroFieldSchema})
(def EnumSymbolToPosKey (s/pair EnumName "enum-name"
                                EnumSymbol "enum-symbol"))
(def EnumPosToSymbolKey (s/pair EnumName "enum-name"
                                EnumSymbolPos "enum-symbol-pos"))
(def EncodeFn (s/=> nil))
(def DecodeFn (s/=> AvroData))
(def Pred (s/=> s/Bool))
(def PathItem (s/conditional
               string? s/Str
               keyword? s/Keyword
               number? s/Num))
(def Path [PathItem])
(def UnionInfo
  {(s/required-key :wrapped?) s/Bool
   (s/required-key :preds) [Pred]
   (s/required-key :name->index) {s/Any s/Int}})
(def Context
  {(s/optional-key :name->schema) SchemaNameToSchema
   (s/optional-key :alias->name) {AvroName AvroName}
   (s/optional-key :enum-symbol->pos) {EnumSymbolToPosKey EnumSymbolPos}
   (s/optional-key :enum-pos->symbol) {EnumPosToSymbolKey EnumSymbol}
   (s/optional-key :record->fields) {RecordName #{FieldName}}
   (s/optional-key :record->field-map) {RecordName FieldMap}
   (s/optional-key :union->union-info) {AvroUnionSchema UnionInfo}})

;;;;;;;;;;;;;;;;;;;; Fns ;;;;;;;;;;;;;;;;;;;;

(defn expand-schema [schema context]
  (let [avro-type (get-avro-type schema)]
    (if (avro-built-in-types avro-type)
      schema
      (-> context :name->schema (get schema)))))

(defn valid-int? [data]
  (and (integer? data)
       (<= data 2147483647)
       (>= data -2147483648)))

(defn valid-long? [data]
  (and (u/long-or-int? data)
       (<= data 9223372036854775807)
       (>= data -9223372036854775808)))

(defn valid-float? [data]
  (and (number? data)
       (<= data 3.4028234E38)
       (>= data -3.4028234E38)))

(defn valid-double? [data]
  (number? data))

(defn valid-bytes-or-string? [data]
  (or (string? data)
      (u/byte-array? data)))

(defn valid-array? [data]
  (sequential? data))

(defn valid-map? [data]
  (and (map? data)
       (if (pos? (count data))
         (string? (-> data first first))
         true)))

(defn valid-record? [data]
  (and (map? data)
       (if (pos? (count data))
         (keyword? (-> data first first))
         true)))

(defn make-pred [schema context]
  (let [type (get-avro-type schema)
        pred (case type
               :null nil?
               :boolean u/boolean?
               :int valid-int?
               :long valid-long?
               :float valid-float?
               :double valid-double?
               :bytes valid-bytes-or-string?
               :string valid-bytes-or-string?
               :array valid-array?
               :map valid-map?
               :enum keyword?
               :fixed #(and (u/byte-array? %)
                            (= (:size schema) (count %)))
               :record valid-record?
               nil)]
    (or pred
        (if-let [schema (-> context :name->schema type)]
          (recur schema context)
          (throw-far-error "Unknown schema type."
                           :illegal-schema :unknown-schema-type
                           (sym-map schema))))))

(defn read-long [is]
  (ms/read-long-varint-zz is))

(defn read-int [is]
  (#?(:clj int
      :cljs #(.toInt %))
   (ms/read-long-varint-zz is)))

(s/defn encode-string :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (valid-bytes-or-string? data)
    (throw-far-error "Data is not a string or bytes."
                     :schema-violation :invalid-string-or-bytes
                     (sym-map data path)))
  (let [s (if (string? data)
            data
            (ru/bytes->utf8-string data))]
    (ms/write-utf8-string os s)))

(s/defn encode-bytes :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (valid-bytes-or-string? data)
    (throw-far-error "Data is not a string or bytes."
                     :schema-violation :invalid-string-or-bytes
                     (sym-map data path)))
  (let [bs (if (string? data)
             (ru/utf8-str->byte-array data)
             data)]
    (ms/write-bytes-w-len-prefix os bs)))

(s/defn encode-double :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (#?(:clj float? :cljs number?) data)
    (throw-far-error "Data is not a floating point number"
                     :schema-violation :invalid-floating-point-number
                     (sym-map data path)))
  (ms/write-double os data))

(s/defn encode-float :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (#?(:clj float? :cljs number?) data)
    (throw-far-error "Data is not a floating point number"
                     :schema-violation :invalid-floating-point-number
                     (sym-map data path)))
  (when (or (> data 3.4028234E38)
            (< data -3.4028234E38))
    (throw-far-error "Value out of range for float"
                     :schema-violation :value-out-of-range-for-float
                     (sym-map data path)))
  (ms/write-float os data))

(s/defn encode-long :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (u/long-or-int? data)
    (throw-far-error "Data is not a long"
                     :schema-violation :invalid-long
                     (sym-map data path)))
  (when (or (> data 9223372036854775807)
            (< data -9223372036854775808))
    (throw-far-error "Value out of range for long"
                     :schema-violation :value-out-of-range-for-long
                     (sym-map data path)))
  (ms/write-long-varint-zz os data))

(s/defn encode-int :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (integer? data)
    (throw-far-error "Data is not an integer"
                     :schema-violation :invalid-integer
                     (sym-map data path)))
  (when (or (> data 2147483647)
            (< data -2147483648))
    (throw-far-error "Value out of range for integer"
                     :schema-violation :value-out-of-range-for-integer
                     (sym-map data path)))
  (ms/write-long-varint-zz os data))

(s/defn encode-boolean :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (ms/write-byte os (case data
                      false 0
                      true 1
                      (throw-far-error "Data is not boolean"
                                       :schema-violation :invalid-boolean
                                       (sym-map data path)))))

(s/defn decode-boolean :- AvroBoolean
  [is :- (s/protocol InputStream)]
  (let [data (ms/read-byte is)]
    (case (int data)
      0 false
      1 true
      (throw-far-error "Invalid data received"
                       :invalid-data :invalid-boolean
                       (sym-map data)))))

(s/defn encode-null :- (s/eq nil)
  "Nulls are encoded as zero bytes."
  [os :- (s/protocol OutputStream)
   data :- AvroData
   path :- Path]
  (when-not (nil? data)
    (throw-far-error "Data is not nil (schema is :null)"
                     :schema-violation :data-not-nil
                     (sym-map data path))))

(s/defn encode-map :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroMapSchema
   context :- Context
   path :- Path]
  (when-not (map? data)
    (throw-far-error "Data is not a map"
                     :schema-violation :invalid-map
                     (sym-map data path)))
  (let [value-schema (:values schema)
        encode-kvp (fn [acc k v]
                     (let [new-path (conj path k)]
                       (when-not (string? k)
                         (throw-far-error "Map keys must be strings"
                                          :schema-violation :invalid-map-key
                                          {:map data :bad-key k :path path}))
                       (encode-string os k new-path)
                       (encode-edn os v value-schema context new-path)))
        num-kvps (count data)]
    (when (pos? num-kvps)
      (encode-long os num-kvps path)
      (reduce-kv encode-kvp nil data))
    (encode-long os 0 path)))

(defn decode-map-block [is m block-count writer-value-schema writer-context]
  (let [decode-kvp (fn [m n]
                     (let [k (ms/read-utf8-string is)
                           v (decode-edn
                              is writer-value-schema writer-context
                              writer-value-schema writer-context)]
                       (assoc m k v)))]
    (reduce decode-kvp m (range block-count))))

(s/defn decode-map :- AvroRecordOrMap
  [is :- (s/protocol InputStream)
   writer-schema :- AvroMapSchema
   writer-context :- Context]
  (let [writer-value-schema (:values writer-schema)
        decode-block (fn [m block-count]
                       (decode-map-block
                        is m block-count writer-value-schema writer-context))
        decode-sized-block (fn [m block-count]
                             (let [num-bytes (read-long is)]
                               (decode-block m block-count)))]
    (loop [m {}]
      (let [block-count (read-long is)]
        (cond
          (pos? block-count) (recur (decode-block m block-count))
          (neg? block-count) (recur (decode-sized-block
                                     m (u/abs block-count)))
          :else m)))))

(s/defn encode-array :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   array :- AvroData
   schema :- AvroArraySchema
   context :- Context
   path :- Path]
  (when-not (sequential? array)
    (throw-far-error "Data is not an array"
                     :schema-violation :invalid-array
                     {:data array :path path}))
  (let [item-schema (:items schema)
        num-items (count array)
        encode-item (fn [index item]
                      (encode-edn os item item-schema context
                                  (conj path index)))]
    (when (pos? num-items)
      (encode-long os num-items path)
      (doall (map-indexed encode-item array)))
    (encode-long os 0 path)))

(defn decode-array-block
  [is array block-count writer-item-schema writer-context]
  (let [decode-item (fn [array n]
                      (let [edn (decode-edn
                                 is writer-item-schema writer-context
                                 writer-item-schema writer-context)]
                        (conj array edn)))]
    (reduce decode-item array (range block-count))))

(s/defn decode-array :- AvroArray
  [is :- (s/protocol InputStream)
   writer-schema :- AvroArraySchema
   writer-context :- Context]
  (let [writer-item-schema (:items writer-schema)
        decode-block (fn [array block-count]
                       (decode-array-block
                        is array block-count writer-item-schema writer-context))
        decode-sized-block (fn [array block-count]
                             (let [num-bytes (read-long is)]
                               (decode-block array block-count)))]
    (loop [array []]
      (let [block-count (read-long is)]
        (cond
          (pos? block-count) (recur (decode-block array block-count))
          (neg? block-count) (recur (decode-sized-block
                                     array (u/abs block-count)))
          :else array)))))

(defn check-named-schemas [writer-schema reader-schema]
  (let [writer-type (get-avro-type writer-schema)
        reader-type (get-avro-type reader-schema)
        writer-name (:name writer-schema)
        reader-name (:name reader-schema)]
    (when-not (= writer-type reader-type)
      (throw-far-error
       (str "Schema mismatch. Writer schema is type " writer-type
            " and reader schema is type " reader-type)
       :schema-mismatch :different-schema-types
       (sym-map writer-schema reader-schema)))
    (when-not (or (= writer-name reader-name)
                  (some #{writer-name} (:aliases reader-schema)))
      (throw-far-error
       (str "Schema mismatch. Reader and writer schemas do not have the "
            "same name. Reader name: `" reader-name
            "`. Writer name: `" writer-name "`")
       :schema-mismatch :different-schema-names
       (sym-map writer-schema reader-schema)))))

(s/defn encode-fixed :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroFixedSchema
   context :- Context
   path :- Path]
  (when-not (byte-array? data)
    (throw-far-error (str "Data for :fixed `" (:name schema)
                          "` is not a fixed byte array")
                     :schema-violation :invalid-fixed
                     (sym-map data schema path)))
  (let [{:keys [size]} schema
        _ (when-not (= (count data) size)
            (throw-far-error "Size of fixed does not match schema"
                             :schema-violation :illegal-size-of-fixed
                             {:schema-size size :data-size (count data)
                              :path path}))]
    (ms/write-bytes os data size)))

(s/defn decode-fixed :- AvroFixedOrBytes
  [is :- (s/protocol InputStream)
   writer-schema :- AvroFixedSchema
   writer-context :- Context]
  (let [writer-size (:size writer-schema)]
    (ms/read-bytes is (int writer-size))))

(s/defn encode-enum :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroEnumSchema
   context :- Context
   path :- Path]
  (let [{:keys [name]} schema
        pos (-> context :enum-symbol->pos (get [name data]))]
    (when-not pos
      (throw-far-error (str "`" data "` is not a valid symbol keyword "
                            "for enum `" (:name schema) "`.")
                       :schema-violation :invalid-enum-symbol
                       (sym-map data schema path)))
    (encode-int os pos path)))

(s/defn decode-enum :- AvroEnum
  [is :- (s/protocol InputStream)
   writer-schema :- AvroEnumSchema
   writer-context :- Context]
  (let [pos (read-long is)]
    (-> writer-context :enum-pos->symbol (get [(:name writer-schema) pos]))))

(s/defn encode-record :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroSchema
   context :- Context
   path :- Path]
  (let [schema (expand-schema schema context)
        _ (when-not (map? data)
            (throw-far-error (str "Data for record `" (:name schema)
                                  "` is not a record.")
                             :schema-violation :invalid-record
                             (sym-map data schema path)))
        {:keys [fields name]} schema
        schema-fields (-> context :record->fields (get name))
        data-fields (set (keys data))
        extra-fields (difference data-fields schema-fields)
        _ (when (seq extra-fields)
            (throw-far-error "Extra record fields found"
                             :schema-violation :extra-record-fields
                             (sym-map extra-fields schema data
                                      path)))
        encode-field (fn [acc field]
                       (let [{:keys [type name default]} field
                             ;; n.b.: We use contains? here because
                             ;; the field value may be nil. We should
                             ;; not use the default if the value is
                             ;; nil, only if the field is missing entirely.
                             field-value (if (contains? data name)
                                           (data name)
                                           default)]
                         (encode-edn os field-value type context
                                     (conj path name))))]
    (reduce encode-field nil fields)))

(defn get-item-schema-info-from-union
  [data schema context]
  (let [{:keys [wrapped? preds]} (-> context :union->union-info (get schema))]
    (loop [union-index 0]
      (when (>= union-index (count schema))
        (throw-far-error "No schemas in union match data"
                         :schema-mismatch :no-schemas-match
                         (sym-map schema data context)))
      (let [pred (preds union-index)]
        (if (pred data)
          (let [item-schema (nth schema union-index)]
            [item-schema wrapped? union-index])
          (recur (inc union-index)))))))

(defn get-unwrapped-union-info
  [data schema context]
  (let [info (get-item-schema-info-from-union data schema context)
        [item-schema _ union-index] info]
    [union-index item-schema data]))

(defn throw-need-wrapping [name data schema unwrapped-data]
  (throw-far-error "Wrapping required for this schema."
                   :illegal-argument :wrapping-required
                   (sym-map name data schema unwrapped-data)))

(defn get-wrapped-union-info
  [data schema context]
  (if (nil? data)
    (get-unwrapped-union-info data schema context)
    (let [[name unwrapped-data] (when (and (map? data)
                                           (= 1 (count data)))
                                  (first data))
          _ (try
              (when (or (not name)
                        (not ((make-pred name context) unwrapped-data)))
                (throw-need-wrapping name data schema unwrapped-data))
              (catch #?(:clj Exception :cljs :default) e
                (throw-need-wrapping name data schema unwrapped-data)))
          union-index (-> context :union->union-info (get schema) :name->index
                          (get name))]
      [union-index name unwrapped-data])))

(s/defn encode-union :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   data :- AvroData
   schema :- AvroUnionSchema
   context :- Context
   path :- Path]
  (let [wrapped? (-> context :union->union-info (get schema) :wrapped?)
        info (if wrapped?
               (get-wrapped-union-info data schema context)
               (get-unwrapped-union-info data schema context))
        [union-index item-schema data] info
        item-schema (expand-schema item-schema context)]
    (ms/write-long-varint-zz os union-index)
    (encode-edn os data item-schema context path)))

(defn get-field-value [m field-name aliases default]
  (let [possible-field-names (concat [field-name] aliases)
        value (-> (keep #(when (contains? m %)
                           (let [field-name %])
                           ;; wrap result in a vector so it will not be
                           ;; nil, even if the field's value is nil.
                           ;; Nil values should not be defaulted, only
                           ;; missing keys.
                           [(m %)])
                        possible-field-names)
                  (first))] ;; Take first match
    (if value
      (first value) ;; Remove vector wrapper
      default)))

(defn resolve-record
  [data writer-schema writer-context reader-schema reader-context]
  (if (= writer-schema reader-schema)
    data
    (let [_ (check-named-schemas writer-schema reader-schema)
          record-name (:name writer-schema) ;; reader has same name
          writer-field-names (-> writer-context :record->fields
                                 (get record-name))
          writer-field-map (-> writer-context :record->field-map
                               (get record-name))
          reader-field-names (-> reader-context :record->fields
                                 (get record-name))
          reader-field-map (-> reader-context :record->field-map
                               (get record-name))
          add-field (fn [acc field-name]
                      (let [reader-field (reader-field-map field-name)
                            {:keys [default aliases]} reader-field
                            field-value (get-field-value
                                         data nil aliases default)]
                        (assoc acc field-name field-value)))
          add-fields (fn [m field-names]
                       (reduce add-field m field-names))]
      (if (= writer-field-names reader-field-names)
        data
        (-> (select-keys data reader-field-names)
            (add-fields (difference reader-field-names
                                    writer-field-names)))))))

(defn resolve-enum
  [written-enum-symbol writer-schema writer-context
   reader-schema reader-context]
  (check-named-schemas writer-schema reader-schema)
  (when-not (-> reader-context :enum-symbol->pos (get [(:name reader-schema)
                                                       written-enum-symbol]))
    (let [reader-enum-symbols (:symbols reader-schema)]
      (throw-far-error
       "Written symbol does not appear in reader enum's symbol set"
       :schema-mismatch :unreadable-enum-symbol
       (sym-map reader-enum-symbols written-enum-symbol writer-schema
                reader-schema))))
  written-enum-symbol)

(defn resolve-array
  [data writer-schema writer-context reader-schema reader-context]
  data)

(defn resolve-map
  [data writer-schema writer-context reader-schema reader-context]
  data)

(defn resolve-fixed [data writer-schema writer-context
                     reader-schema reader-context]
  (check-named-schemas writer-schema reader-schema)
  (let [writer-size (:size writer-schema)
        reader-size (:size reader-schema)]
    (when-not (= writer-size reader-size)
      (throw-far-error
       (str "Schema mismatch. Writer size is " writer-size
            " and reader size is " reader-size)
       :schema-mismatch :fixed-size-mismatch
       (sym-map writer-schema reader-schema))))
  data)

(defn throw-primitive-schema-mismatch [writer-schema reader-schema]
  (throw-far-error
   (str "Schemas are not compatible. Writer schema is `" writer-schema
        "` and reader schema is `" reader-schema "`")
   :schema-mismatch :incompatible-primitives
   (sym-map writer-schema reader-schema)))

(defn resolve-null [data writer-schema reader-schema]
  (when-not (= :null (get-avro-type reader-schema))
    (throw-primitive-schema-mismatch writer-schema reader-schema))
  nil)

(defn resolve-boolean [data writer-schema reader-schema]
  (when-not (= :boolean (get-avro-type reader-schema))
    (throw-primitive-schema-mismatch writer-schema reader-schema))
  data)

(defn resolve-int [data writer-schema reader-schema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :int #?(:clj int :cljs identity)
                  :long u/long
                  :float #?(:clj float :cljs identity)
                  :double #?(:clj double :cljs identity)
                  (throw-primitive-schema-mismatch writer-schema
                                                   reader-schema))]
    (cast-fn data)))

(defn resolve-long [data writer-schema reader-schema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :long u/long
                  :float float
                  :double double
                  (throw-primitive-schema-mismatch writer-schema
                                                   reader-schema))]
    (cast-fn data)))

(defn resolve-float [data writer-schema reader-schema]
  (let [cast-fn (case (get-avro-type reader-schema)
                  :float float
                  :double double
                  (throw-primitive-schema-mismatch writer-schema
                                                   reader-schema))]
    (cast-fn data)))

(defn resolve-double [data writer-schema reader-schema]
  (when-not (= :double (get-avro-type reader-schema))
    (throw-primitive-schema-mismatch writer-schema reader-schema))
  data)

(defn resolve-bytes [data writer-schema reader-schema]
  (case (get-avro-type reader-schema)
    :bytes data
    :string (ru/bytes->utf8-string data)
    (throw-primitive-schema-mismatch writer-schema reader-schema)))

(defn resolve-string [data writer-schema reader-schema]
  (case (get-avro-type reader-schema)
    :bytes (ru/utf8-str->byte-array data)
    :string data
    (throw-primitive-schema-mismatch writer-schema reader-schema)))

(defn wrap [data wrapped? reader-schema]
  (if (or (nil? data)
          (not wrapped?))
    data
    (let [name (ru/get-schema-name reader-schema)]
      {name data})))

(defn resolve
  [data writer-schema writer-context reader-schema reader-context]
  (let [info (if (union? reader-schema)
               (get-item-schema-info-from-union data reader-schema
                                                reader-context)
               [reader-schema false])
        [reader-schema reader-wrapped?] info
        writer-type (ru/get-avro-type writer-schema)
        resolver (case writer-type
                   :record resolve-record
                   :enum resolve-enum
                   :array resolve-array
                   :map resolve-map
                   :fixed resolve-fixed
                   :null resolve-null
                   :boolean resolve-boolean
                   :int resolve-int
                   :long resolve-long
                   :float resolve-float
                   :double resolve-double
                   :bytes resolve-bytes
                   :string resolve-string)
        ret (if (= reader-schema writer-schema)
              data
              (if (avro-primitive-types writer-type)
                (resolver data writer-schema reader-schema)
                (resolver data writer-schema writer-context reader-schema
                          reader-context)))]
    (wrap ret reader-wrapped? reader-schema)))

(s/defn decode-record :- AvroRecordOrMap
  [is :- (s/protocol InputStream)
   writer-schema :- AvroRecordSchema
   writer-context :- Context]
  (let [add-field (fn [rec field]
                    (let [{:keys [name type]} field
                          field-schema (expand-schema type writer-context)
                          value  (decode-edn is field-schema writer-context
                                             field-schema writer-context)]
                      (assoc rec name value)))]
    (reduce add-field {} (:fields writer-schema))))

(s/defn encode-edn :- (s/eq nil)
  [os :- (s/protocol OutputStream)
   edn :- AvroData
   schema :- AvroSchema
   context :- Context
   path :- Path]
  (let [avro-type (get-avro-type (expand-schema schema context))
        encoder (case avro-type
                  :record encode-record
                  :map encode-map
                  :enum encode-enum
                  :array encode-array
                  :fixed encode-fixed
                  :union encode-union
                  :null encode-null
                  :boolean encode-boolean
                  :int encode-int
                  :long encode-long
                  :float encode-float
                  :double encode-double
                  :bytes encode-bytes
                  :string encode-string)]
    (if (avro-primitive-types avro-type)
      (encoder os edn path)
      (encoder os edn schema context path)))
  (let [num-bytes (ms/get-size os)]
    (when (> num-bytes max-bytes-per-avro-obj)
      (throw-far-error (str "Serialized data exceeds maximum size limit of "
                            max-bytes-per-avro-obj " bytes")
                       :runtime-error :serialized-data-too-large
                       (sym-map max-bytes-per-avro-obj num-bytes)))))

(def type->matching-types
  {:int #{:int :long :float :double}
   :long #{:long :float :double}
   :float #{:float :double}
   :double #{:double}
   :bytes #{:bytes :string}
   :string #{:bytes :string}})

(defn get-reader-item-schema-from-union
  [writer-name reader-schema]
  (let [match-set (or (type->matching-types writer-name)
                      #{writer-name})]
    (-> (keep-indexed (fn [index item-schema]
                        (when (match-set (ru/get-schema-name item-schema))
                          (nth reader-schema index)))
                      reader-schema)
        (first))))

(defn get-writer-name [writer-item-schema reader-context]
  (let [name (ru/get-schema-name writer-item-schema)]
    (or (-> reader-context :aliases name)
        name)))

(defn decode-union
  [is writer-schema writer-context reader-schema reader-context]
  (let [union-index (#?(:cljs .toInt :clj identity) (read-long is))
        writer-item-schema (nth writer-schema union-index)
        writer-name (get-writer-name writer-item-schema reader-context)
        reader-item-schema (if (union? reader-schema)
                             (get-reader-item-schema-from-union
                              writer-name reader-schema)
                             reader-schema)
        data (decode-edn is writer-item-schema writer-context
                         reader-item-schema reader-context)]
    (if-not (union? reader-schema)
      data
      (let [wrapped? (-> reader-context :union->union-info
                         (get reader-schema) :wrapped?)]
        (wrap data wrapped? reader-item-schema)))))

(s/defn decode-edn :- AvroData
  [is :- (s/protocol InputStream)
   writer-schema :- AvroSchema
   writer-context :- Context
   reader-schema :- AvroSchema
   reader-context :- Context]
  (if (union? writer-schema)
    (decode-union is writer-schema writer-context reader-schema reader-context)
    (let [writer-schema (expand-schema writer-schema writer-context)
          reader-schema (expand-schema reader-schema reader-context)]
      (let [writer-avro-type (get-avro-type writer-schema)
            dec-fn (case writer-avro-type
                     :record decode-record
                     :enum decode-enum
                     :array decode-array
                     :map decode-map
                     :fixed decode-fixed
                     :null (constantly nil) ;; Nulls are encoded as zero bytes
                     :boolean decode-boolean
                     :int read-int
                     :long read-long
                     :float ms/read-float
                     :double ms/read-double
                     :bytes ms/read-len-prefixed-bytes
                     :string ms/read-utf8-string)
            data (if (avro-primitive-types writer-avro-type)
                   (dec-fn is)
                   (dec-fn is writer-schema writer-context))]
        (resolve data writer-schema writer-context reader-schema
                 reader-context)))))

(defn get-child-schemas [schema avro-type]
  (case avro-type
    :record (map :type (:fields schema))
    :array [(:items schema)]
    :map [(:values schema)]
    :union schema
    []))

(defn add-child-schemas [context schema avro-type]
  (let [child-schemas (get-child-schemas schema avro-type)
        add-child (fn [context child-schema]
                    (when (and (= :union avro-type)
                               (= :union (get-avro-type child-schema)))
                      (throw-far-error
                       "Unions may not immediately contain other unions"
                       :illegal-schema :illegal-union-nesting
                       (sym-map schema child-schema context)))
                    (add-schema context child-schema))]
    (reduce add-child context child-schemas)))

(defn add-name [context [name schema]]
  (when (nil? name)
    (let [{:keys [type]} schema]
      (throw-far-error (str "Missing schema name. " type " schemas "
                            "must have a name.")
                       :illegal-schema :missing-schema-name
                       (sym-map schema type name))))
  (let [{:keys [name->schema]} context]
    (when-not (valid-avro-name? name)
      (throw-far-error (str "`" name "` is not a valid name keyword. "
                            "Names must be keywords whose names match this "
                            "regex: " avro-name-pattern-str " to be valid")
                       :illegal-schema :illegal-avro-name
                       (sym-map name schema)))
    (when (and name->schema
               (name->schema name)
               (not= schema (name->schema name)))
      (throw-far-error (str "Different schemas have the same name: `" name)
                       :illegal-schema :same-name-for-different-schemas
                       (sym-map name schema context)))
    (assoc-in context [:name->schema name] schema)))


(defn add-enum-info [context name schema]
  (let [{:keys [symbols]} schema
        f (fn [context [pos symbol]]
            (when-not (valid-avro-name? symbol)
              (throw-far-error
               (str "Illegal enum symbol `" symbol "` in schema. Symbols must "
                    "be keywords whose names match this regex: "
                    avro-name-pattern-str " to be valid")
               :illegal-schema :illegal-enum-symbol
               (sym-map schema symbol)))
            (-> context
                (assoc-in [:enum-symbol->pos [name symbol]] pos)
                (assoc-in [:enum-pos->symbol [name pos]] symbol)))]
    (when-not (seq symbols)
      (throw-far-error "Illegal enum schema. Symbol set is empty"
                       :illegal-schema :empty-enum-symbol-set
                       (sym-map schema)))
    (when-not (sequential? symbols)
      (throw-far-error "Illegal enum schema. Symbol set is not sequential."
                       :illegal-schema :symbol-set-not-sequential
                       (sym-map schema)))
    (when-not (apply distinct? symbols)
      (throw-far-error "Duplicates present in enum symbols list"
                       :illegal-schema :duplicate-enum-symbols
                       (sym-map symbols name schema)))
    (reduce f context (map-indexed vector symbols))))

(defn check-record-info [record-name schema field-names field]
  (let [field-name (:name field)]
    (when (and field-names
               (field-names field-name))
      (throw-far-error
       (str "Duplicate field name `" field-name
            "` in record schema")
       :illegal-schema :duplicate-field-name
       (sym-map record-name schema field-name)))
    (when-not (valid-avro-name? field-name)
      (throw-far-error
       (str "Illegal field name `" field-name
            "` in record schema. Field names must be "
            "keywords whose names match this regex: "
            avro-name-pattern-str " to be valid")
       :illegal-schema :illegal-record-field-name
       (sym-map schema record-name field-name)))
    ;; We use contains? here because the default value can be nil
    (when-not (contains? field :default)
      (throw-far-error
       (str "Field `" field-name "` does not have a default value")
       :illegal-schema :missing-default
       (sym-map schema field-name)))))

(defn update-aliases [context record-name aliases field]
  (let [update-alias (fn [context alias-name]
                       (assoc-in context
                                 [:record->field-map record-name alias-name]
                                 field))]
    (reduce update-alias context aliases)))

(defn add-record-info [context record-name schema]
  (let [add-field (fn [context field]
                    (let [{:keys [name aliases]} field
                          field-type (:type field)
                          field-names (-> context :record->fields record-name)]
                      (check-record-info record-name schema
                                         field-names field)
                      (-> context
                          (update-in [:record->fields record-name]
                                     #(conj (or % #{}) name))
                          (assoc-in [:record->field-map record-name name]
                                    field)
                          (update-aliases record-name aliases field))))]
    (reduce add-field context (:fields schema))))

(defn add-enum-and-record-info [context]
  (let [{:keys [name->schema]} context
        f (fn [context name schema]
            (case (get-avro-type schema)
              :enum (add-enum-info context name schema)
              :record (add-record-info context name schema)
              context))]
    (reduce-kv f context name->schema)))

(defn add-named-schema [context schema]
  (let [{:keys [name aliases]} schema
        all-names (conj aliases name)
        name-schema-pairs (map #(vector % schema) all-names)
        context (reduce (fn [acc alias]
                          (assoc-in acc [:alias->name alias] name))
                        context aliases)]
    (reduce add-name context name-schema-pairs)))

(defn check-union-schema [schema]
  (when-not (apply distinct? schema)
    (throw-far-error "Illegal union schema. Members are not unique."
                     :illegal-schema :non-unique-union-members
                     (sym-map schema)))
  (when (pos? (ru/count-types schema #{:union}))
    (throw-far-error "Unions may not immediately contain other unions."
                     :illegal-schema :illegal-union-nesting
                     (sym-map schema)))
  (when (ru/multiple-types? schema #{:array})
    (throw-far-error "Unions may only contain one array type."
                     :illegal-schema :too-many-array-types
                     (sym-map schema)))
  (when (ru/multiple-types? schema #{:map})
    (throw-far-error "Unions may only contain one map type."
                     :illegal-schema :too-many-map-types
                     (sym-map schema))))

(defn make-union-info [context schema]
  (check-union-schema schema)
  (let [wrapped? (ru/union-wrap-reqd? schema)
        preds (mapv #(make-pred % context) schema)
        make-name-map (fn [index sub-schema]
                        {(ru/get-schema-name sub-schema) index})
        name->index (->> (map-indexed make-name-map schema)
                         (apply merge))]
    (sym-map wrapped? preds name->index)))

(defn add-union-schema [context schema]
  (let [{:keys [union->union-info]} context]
    (if (and union->union-info
             (union->union-info schema))
      context
      (assoc-in context [:union->union-info schema]
                (make-union-info context schema)))))

(defn add-schema [context schema]
  (let [avro-type (get-avro-type schema)
        named? (avro-named-types avro-type)
        recursive? (avro-recursive-types avro-type)
        union? (= :union avro-type)
        {:keys [name->schema]} context]
    (when-not (or
               (avro-primitive-types avro-type)
               named?
               recursive?
               (and name->schema
                    (name->schema avro-type)))
      (throw-far-error (str "Unknown schema type `" avro-type "`.")
                       :illegal-schema :unknown-schema-type
                       (sym-map schema)))

    (cond-> context
      union? (add-union-schema schema)
      named? (add-named-schema schema)
      recursive? (add-child-schemas schema avro-type))))

(defn check-record-field-defaults [context]
  (doseq [[record-name fields-map] (:record->field-map context)]
    (doseq [field (vals fields-map)]
      (let [{:keys [type default]} field
            avro-type (get-avro-type type)
            field-type (if (= :union avro-type)
                         (first type)
                         type)]
        (try
          (let [os (ios/make-mutable-output-stream)]
            (encode-edn os default field-type context []))
          (catch #?(:clj clojure.lang.ExceptionInfo
                    :cljs cljs.core.ExceptionInfo) e
            (if (= :schema-violation (:type (ex-data e)))
              (throw-far-error
               (str "The default value `" default "` for field `" (:name field)
                    "` is not of type `" field-type
                    (if (= :union avro-type)
                      (str "`. Default values for union fields must match "
                           "the type of the first element of the union.")
                      "`."))
               :schema-violation :incorrect-default-type
               (sym-map record-name field type default))
              (throw e)))))))
  context)

(s/defn make-context-impl :- Context
  [top-schema :- AvroSchema]
  (-> (add-schema {} top-schema)
      (add-enum-and-record-info)
      (check-record-field-defaults)))

(def context-cache (stock/make-stockroom default-context-cache-size))

(defn make-context
  [top-schema]
  (if-let [context (stock/get context-cache top-schema)]
    context
    (let [context (make-context-impl top-schema)]
      (stock/put context-cache top-schema context)
      context)))

(s/defn check-schema :- nil
  [schema :- AvroSchema]
  ;; make-context does schema checking, throwing :illegal-schema as needed
  (make-context schema)
  nil)

(s/defn make-default-record :- AvroData
  [schema :- AvroRecordSchema]
  (check-schema schema)
  (when-not (= :record (:type schema))
    (throw-far-error "make-default-record can only be called on record schemas"
                     :illegal-argument :not-a-record-schema
                     (sym-map schema)))
  (let [add-field (fn [acc {:keys [type name default]}]
                    (assoc acc name (if (= :record (get-avro-type type))
                                      (make-default-record type)
                                      default)))]
    (reduce add-field {} (:fields schema))))

(s/defn valid-schema? :- s/Bool
  [schema :- AvroSchema]
  (try
    (check-schema schema)
    true
    (catch ExceptionInfo e
      (if (= :illegal-schema (:type (ex-data e)))
        false
        (throw e)))))

(defn edn->output-stream [schema data]
  (let [context (make-context schema)
        os (ios/make-mutable-output-stream)]
    (try
      (encode-edn os data schema context [])
      (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
        (let [orig-error-map (ex-data e)
              {:keys [type subtype]} orig-error-map
              orig-stack-trace (u/get-exception-stacktrace e)
              orig-msg (u/get-exception-msg e)
              top-level-schema schema
              top-level-data data]
          (throw-far-error "Encode error"
                           type subtype
                           (merge
                            orig-error-map
                            (sym-map top-level-schema top-level-data
                                     orig-stack-trace orig-msg))))))
    os))

(defn input-stream->edn [input-stream writer-schema reader-schema]
  (let [writer-context (make-context writer-schema)
        reader-context (make-context reader-schema)
        edn (decode-edn input-stream writer-schema writer-context
                        reader-schema reader-context)
        remaining-count (ms/get-available-count input-stream)]
    (when-not (zero? remaining-count)
      (throw-far-error "Extra items in stream"
                       :illegal-argument :extra-items-in-stream
                       (sym-map edn remaining-count
                                reader-schema writer-schema)))
    edn))

(s/defn edn->avro-byte-array :- AvroFixedOrBytes
  [schema :- s/Any ;; AvroSchema
   edn :- AvroData]
  (-> (edn->output-stream schema edn)
      (ms/to-byte-array)))

(s/defn edn->avro-b64-string :- s/Str
  [schema :- AvroSchema
   edn :- AvroData]
  (-> (edn->output-stream schema edn)
      (ms/to-b64-string)))

(defn handle-decode-error [e avro-b64-string writer-schema reader-schema]
  (let [orig-error-map (ex-data e)
        orig-stack-trace (u/get-exception-stacktrace e)
        orig-msg (u/get-exception-msg e)]
    (throw-far-error "Decode error"
                     :illegal-argument :decode-error
                     (sym-map writer-schema reader-schema avro-b64-string
                              orig-error-map orig-stack-trace orig-msg))))

(s/defn avro-byte-array->edn :- AvroData
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-byte-array :- AvroFixedOrBytes]
  (try
    (-> (ios/byte-array->mutable-input-stream avro-byte-array)
        (input-stream->edn writer-schema reader-schema))
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
      (let [avro-b64-string (u/byte-array->b64 avro-byte-array)]
        (handle-decode-error e avro-b64-string writer-schema reader-schema)))))

(s/defn avro-b64-string->edn :- AvroData
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-b64-string :- s/Str]
  (try
    (-> (ios/b64-string->mutable-input-stream avro-b64-string)
        (input-stream->edn writer-schema reader-schema))
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
      (handle-decode-error e avro-b64-string writer-schema reader-schema))))

(s/defn set-context-cache-size! :- nil
  [num-keys :- s/Int]
  (stock/set-num-keys! context-cache num-keys))

(s/defn get-cur-context-cache-size :- s/Int
  []
  (stock/get-cur-num-keys context-cache))
