(ns farbetter.roe.prismatic
  (:require
   [farbetter.roe.schemas :refer
    [AvroFixedOrBytes AvroLong AvroSchema avro-primitive-types]]
   [farbetter.roe.serdes :as serdes]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare edn-schema->prismatic-schema xf-array xf-enum xf-fixed xf-map
         xf-record)

(def LongOrNum (s/if u/long?
                 AvroLong
                 s/Num))

(defn primitive->test [primitive]
  (case primitive
    :null nil?
    :boolean #(or (true? %)
                  (false? %))
    :int integer?
    :long u/long?
    :float number?
    :double number?
    :bytes u/byte-array?
    :string string?))

(defn xf-primitive [primitive]
  (case primitive
    :null (s/eq nil)
    :boolean s/Bool
    :int s/Int
    :long LongOrNum
    :float s/Num
    :double s/Num
    :bytes AvroFixedOrBytes
    :string s/Str))

(defn schema->conditionals [schema]
  (cond
    (contains? avro-primitive-types schema)
    [(primitive->test schema) (xf-primitive schema)]

    (contains? avro-primitive-types (:type schema))
    (let [primitive (:type schema)]
      [(primitive->test primitive) (xf-primitive primitive)])

    (= :map (:type schema))
    [map? (xf-map schema)]

    (= :record (:type schema))
    [map? (xf-record schema)]

    (= :enum (:type schema))
    [#(some #{%} (:symbols schema)) (xf-enum schema)]

    (= :array (:type schema))
    [sequential? (xf-array schema)]

    (= :fixed (:type schema))
    [u/byte-array? (xf-fixed schema)]

    (keyword? schema)
    ;; TODO: Implement
    (throw-far-error "Transformation of named references is not yet implemented"
                     :execution-error :not-implemented (sym-map schema))

    :else (throw-far-error "Schema is not a valid Avro EDN schema"
                           :illegal-argument :illegal-schema
                           (sym-map schema))))

(defn xf-union [union-schema]
  (let [add-schema (fn [acc schema]
                     (concat acc (schema->conditionals schema)))
        arglist (reduce add-schema [] union-schema)]
    (apply s/conditional arglist)))

(defn xf-primitive-map
  [schema]
  (xf-primitive (:type schema)))

(defn xf-record
  [schema]
  (let [add-field (fn [acc {:keys [name type]}]
                    (if (and
                         (sequential? type)
                         (= :null (first type)))
                      (let [xf-type (if (= 2 (count type))
                                      (edn-schema->prismatic-schema
                                       (second type))
                                      (xf-union (rest type)))]
                        (assoc acc (s/optional-key name) (s/maybe xf-type)))
                      (let [xf-type (edn-schema->prismatic-schema type)]
                        (assoc acc (s/required-key name) xf-type))))]
    (reduce add-field {} (:fields schema))))

(defn xf-enum
  [{:keys [symbols]}]
  (apply s/enum symbols))

(defn xf-array
  [schema]
  (let [items-schema (edn-schema->prismatic-schema (:items schema))]
    [items-schema]))

(defn xf-map
  [schema]
  (let [vals-schema (edn-schema->prismatic-schema (:values schema))]
    {s/Str vals-schema}))

(defn xf-fixed
  [schema]
  AvroFixedOrBytes)

(defn xf-named-reference
  [schema]
  ;; TODO: Implement
  (throw-far-error "Transformation of named references is not yet implemented"
                   :execution-error :not-implemented (sym-map schema)))

(defn edn-schema->prismatic-schema [schema]
  (serdes/check-schema schema)
  (cond
    (sequential? schema) (xf-union schema)
    (contains? avro-primitive-types schema) (xf-primitive schema)
    (contains? avro-primitive-types (:type schema)) (xf-primitive-map schema)
    (= :record (:type schema)) (xf-record schema)
    (= :enum (:type schema)) (xf-enum schema)
    (= :array (:type schema)) (xf-array schema)
    (= :map (:type schema)) (xf-map schema)
    (= :fixed (:type schema)) (xf-fixed schema)
    (keyword? schema) (xf-named-reference schema)
    :else (throw-far-error "Schema is not a valid Avro EDN schema"
                           :illegal-argument :illegal-schema
                           (sym-map schema))))
