(ns farbetter.roe
  (:require
   [farbetter.roe.bencode :as bc]
   [farbetter.roe.fingerprint :as f]
   [farbetter.roe.json :as json]
   [farbetter.roe.prismatic :as prismatic]
   [farbetter.roe.schemas :refer [AvroData AvroFixedOrBytes AvroSchema]]
   [farbetter.roe.serdes :as serdes]
   [farbetter.utils :as u :refer [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(def LongType (s/pred u/long?))
(def PrismaticSchema s/Any)

(defn check-for-nil-schema [schema]
  (when (nil? schema)
    (throw-far-error "Schema is nil."
                     :illegal-schema :schema-is-nil
                     (sym-map schema))))

(s/defn edn->avro-byte-array :- AvroFixedOrBytes
  "Encodes an EDN data structure into an Avro byte array."
  [schema :- AvroSchema
   edn :- AvroData]
  (check-for-nil-schema schema)
  (serdes/edn->avro-byte-array schema edn))

(s/defn avro-byte-array->edn :- AvroData
  "Decodes an Avro byte array into an EDN data structure."
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-byte-array :- AvroFixedOrBytes]
  (check-for-nil-schema writer-schema)
  (check-for-nil-schema reader-schema)
  (when (nil? avro-byte-array)
    (throw-far-error "avro-byte-array is nil"
                     :illegal-argument :avro-byte-array-is-nil
                     (sym-map avro-byte-array)))
  (serdes/avro-byte-array->edn writer-schema reader-schema avro-byte-array))

(s/defn edn->avro-b64-string :- s/Str
  "Encodes a Clojure data structure into an Avro base-64 string."
  [schema :- AvroSchema
   edn :- AvroData]
  (check-for-nil-schema schema)
  (serdes/edn->avro-b64-string schema edn))

(s/defn avro-b64-string->edn :- AvroData
  "Decodes an Avro base-64 string into a Clojure data structure."
  [writer-schema :- AvroSchema
   reader-schema :- AvroSchema
   avro-str :- s/Str]
  (check-for-nil-schema writer-schema)
  (check-for-nil-schema reader-schema)
  (when (nil? avro-str)
    (throw-far-error "avro-str is nil"
                     :illegal-argument :avro-str-is-nil
                     (sym-map avro-str)))
  (serdes/avro-b64-string->edn writer-schema reader-schema avro-str))

(s/defn check-schema :- nil
  "Checks schema for validity. Throws :illegal-schema exceptions if invalid."
  [schema :- AvroSchema]
  (check-for-nil-schema schema)
  (serdes/check-schema schema)
  nil)

(s/defn valid-schema? :- s/Bool
  "Checks schema for validity. Returns true or false."
  [schema :- AvroSchema]
  (if (nil? schema)
    false
    (serdes/valid-schema? schema)))

(s/defn edn-schema->fingerprint :- LongType
  "Returns a map of two ints representing the 64-bit Rabin fingerprint
   of the schema."
  [schema :- AvroSchema]
  (check-for-nil-schema schema)
  (let [bencoded-canonical-form (bc/edn-schema->bencode schema true)]
    (f/fingerprint64 bencoded-canonical-form)))

(s/defn json-schema->edn-schema :- AvroSchema
  "Translates a JSON Avro schema into an EDN Avro schema."
  [json-schema :- s/Str]
  (check-for-nil-schema json-schema)
  (when (clojure.string/blank? json-schema)
    (throw-far-error "json-schema is empty"
                     :illegal-schema :schema-is-blank
                     (sym-map json-schema)))
  (json/json->edn-schema json-schema))

(s/defn edn-schema->json-schema :- s/Str
  "Translates an EDN Avro schema into a JSON Avro schema."
  [schema :- AvroSchema]
  (check-for-nil-schema schema)
  (json/edn-schema->json schema))

(s/defn edn-schema->prismatic-schema :- PrismaticSchema
  "Translates an EDN Avro schema into a PrismaticSchema"
  [schema :- AvroSchema]
  (check-for-nil-schema schema)
  (prismatic/edn-schema->prismatic-schema schema))

(s/defn make-default-record :- AvroData
  "Makes a default record for the given record schema."
  [schema :- AvroSchema]
  (check-for-nil-schema schema)
  (serdes/make-default-record schema))
