(ns farbetter.roe.io-streams
  (:require
   [farbetter.roe.mutable-streams :as ms]
   [farbetter.utils :as u :refer [throw-far-error inspect sym-map]])
  (:import
   [com.google.common.io LittleEndianDataInputStream
    LittleEndianDataOutputStream]
   [java.io ByteArrayInputStream ByteArrayOutputStream]
   [java.math BigInteger]
   [java.nio ByteBuffer ByteOrder]
   [java.util Base64]))

(defrecord MutableOutputStream [baos ledos]
  ms/OutputStream
  (get-size [this]
    (.size ^ByteArrayOutputStream baos))

  (write-byte [this b]
    (.writeByte ^LittleEndianDataOutputStream ledos b))

  (write-bytes [this bs num-bytes]
    (.write ^LittleEndianDataOutputStream ledos bs 0 num-bytes))

  (write-bytes-w-len-prefix [this bs]
    (let [num-bytes (count bs)]
      (ms/write-long-varint-zz this num-bytes)
      (ms/write-bytes this bs num-bytes)))

  (write-utf8-string [this s]
    (let [bytes (.getBytes ^String s "UTF-8")
          num-bytes (count bytes)]
      (ms/write-long-varint-zz this num-bytes)
      (ms/write-bytes this bytes num-bytes)))

  (write-long-varint-zz [this l]
    (let [l (cond-> l
              (instance? BigInteger l) (.longValue))
          zz-n (bit-xor (bit-shift-left l 1) (bit-shift-right l 63))]
      (loop [n zz-n]
        (if (zero? (bit-and n -128))
          (let [b (bit-and n 0x7f)]
            (.writeByte ^LittleEndianDataOutputStream ledos b))
          (let [b (-> (bit-and n 0x7f)
                      (bit-or 0x80))]
            (.writeByte ^LittleEndianDataOutputStream ledos b)
            (recur (unsigned-bit-shift-right n 7)))))))

  (write-float [this f]
    (.writeFloat ^LittleEndianDataOutputStream ledos ^float f))

  (write-double [this d]
    (.writeDouble ^LittleEndianDataOutputStream ledos ^double d))

  (to-byte-array [this]
    (.toByteArray ^ByteArrayOutputStream baos))

  (to-b64-string [this]
    (let [b64-encoder (Base64/getEncoder)]
      (->> (.toByteArray ^ByteArrayOutputStream baos)
           (.encodeToString b64-encoder))))

  (to-utf8-string [this]
    (String. #^bytes (ms/to-byte-array this) "UTF-8")))

(defrecord MutableInputStream [bais ledis]
  ms/InputStream
  (get-available-count [this]
    (.available ^ByteArrayInputStream bais))

  (read-byte [this]
    (.readByte ^LittleEndianDataInputStream ledis))

  (read-bytes [this num-bytes]
    (let [ba (byte-array num-bytes)]
      (.readFully ^LittleEndianDataInputStream ledis #^bytes ba 0 num-bytes)
      ba))

  (read-len-prefixed-bytes [this]
    (let [num-bytes (ms/read-long-varint-zz this)]
      (ms/read-bytes this num-bytes)))

  (read-utf8-string [this]
    (String. #^bytes (ms/read-len-prefixed-bytes this) "UTF-8"))

  (read-long-varint-zz [this]
    (loop [i 0
           out 0]
      (let [b (.readByte ^LittleEndianDataInputStream ledis)]
        (if (zero? (bit-and b 0x80))
          (let [zz-n (-> (bit-shift-left b i)
                         (bit-or out))
                long-out (->> (bit-and zz-n 1)
                              (- 0)
                              (bit-xor (unsigned-bit-shift-right zz-n 1)))]
            long-out)
          (let [out (-> (bit-and b 0x7f)
                        (bit-shift-left i)
                        (bit-or out))
                i (+ 7 i)]
            (if (<= i 63)
              (recur i out)
              (throw-far-error "Variable-length quantity is more than 64 bits"
                               :invalid-data :var-len-num-more-than-64-bits
                               (sym-map i))))))))

  (read-float [this]
    (.readFloat ^LittleEndianDataInputStream ledis))

  (read-double [this]
    (.readDouble ^LittleEndianDataInputStream ledis))

  (mark [this read-limit]
    (.mark ^LittleEndianDataInputStream ledis ^int read-limit))

  (reset [this]
    (.reset ^ByteArrayInputStream bais)))

(defn byte-array->mutable-input-stream [bs]
  (let [bais (ByteArrayInputStream. bs)
        ledis (LittleEndianDataInputStream. bais)]
    (->MutableInputStream bais ledis)))

(defn b64-string->mutable-input-stream [s]
  (let [b64-decoder (Base64/getDecoder)
        bs (.decode b64-decoder ^String s)
        bais (ByteArrayInputStream. bs)
        ledis (LittleEndianDataInputStream. bais)]
    (->MutableInputStream bais ledis)))

(defn make-mutable-output-stream []
  (let [baos (ByteArrayOutputStream.)
        ledos (LittleEndianDataOutputStream. baos)]
    (->MutableOutputStream baos ledos)))
