(ns farbetter.roe.bencode
  (:require
   [clojure.string :as string :refer [join]]
   [farbetter.roe.schemas :refer [AvroSchema]]
   [farbetter.roe.transform :refer [transform-schema]]
   [farbetter.roe.utils :refer [#?@(:cljs [float?])]]
   [farbetter.utils :as u :refer
    [byte-array? long? throw-far-error
     #?@(:clj [inspect sym-map] :cljs [long])]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

;; This implementation follows the bencoding rules at:
;; https://wiki.theory.org/BitTorrentSpecification#Bencoding
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declare bencode->obj* obj->bencode* parse-obj)

(def digits #{"9" "3" "4" "8" "7" "5" "6" "1" "0" "2"})

;;;;;;;;; API ;;;;;;;;;

(s/defn obj->bencode :- s/Str
  [obj :- s/Any]
  (obj->bencode* obj))

(s/defn bencode->obj :- s/Any
  [s :- s/Str]
  (bencode->obj* s))

(s/defn ^:always-validate edn-schema->bencode :- s/Str
  ([schema :- AvroSchema]
   (edn-schema->bencode schema false))
  ([schema :- AvroSchema
    canonicalize? :- s/Bool]
   (let [transformed-schema (transform-schema schema :encode canonicalize?)]
     (obj->bencode transformed-schema))))

(s/defn bencode->edn-schema :- AvroSchema
  [s :- s/Str]
  (-> (bencode->obj s)
      (transform-schema :decode false)))

;;;;;;;;; Helper fns ;;;;;;;;;

(defn enc-int [i]
  (if (and (<= i 2147483647) (>= i -2147483648))
    (str "i" i "e")
    (throw-far-error "Value out of range for integer"
                     :illegal-argument :not-an-integer {:value i})))

(defn enc-str [s]
  (str (count s) ":" s))

(defn enc-list [l]
  (let [items (join (map obj->bencode l))]
    (str "l" items "e")))

(defn enc-kv [[k v]]
  (if (string? k)
    (-> (enc-str k)
        (str (obj->bencode v)))
    (throw-far-error
     "Illegal non-string key in map type"
     :illegal-argument :non-string-key
     (sym-map k v))))

(defn enc-map [m]
  (let [kvs (seq (into (sorted-map) m))
        enc-kvs (join (map enc-kv kvs))]
    (str "d" enc-kvs "e")))

(defn obj->bencode* [obj]
  (let [enc-fn (cond
                 (integer? obj) enc-int
                 (string? obj) enc-str
                 (sequential? obj) enc-list
                 (map? obj) enc-map
                 :else #(throw-far-error "Unsupported type."
                                         :illegal-argument :unsupported-type
                                         {:value %}))]
    (enc-fn obj)))

(defn- str->int [s]
  (let [i #?(:clj
             (try
               (Integer/parseInt s)
               (catch Exception e
                 :not-an-integer))
             :cljs
             (let [i (js/parseInt s)]
               (if (js/isNaN i)
                 :not-an-integer
                 i)))]
    (when (= :not-an-integer i)
      (throw-far-error "Not an integer"
                       :illegal-argument :not-an-integer
                       (sym-map s)))
    i))

(defn find-pos [s ch]
  (reduce (fn [acc c]
            (if (= ch (str c))
              (reduced acc)
              (inc acc)))
          0 s))

(defn parse-int [s]
  (let [end-pos (find-pos s "e")
        val (str->int (subs s 1 end-pos))]
    [val (subs s (inc end-pos))]))

(defn parse-str [s]
  (let [colon-pos (find-pos s ":")
        len (str->int (subs s 0 colon-pos))
        start-pos (inc colon-pos)
        end-pos (+ start-pos len)
        val (subs s start-pos end-pos)]
    [val (subs s end-pos)]))

(defn parse-list [bencoded-str]
  (loop [s (subs bencoded-str 1)
         output []]
    (cond
      (= "e" (str (first s))) [output (subs s 1)]
      (zero? (count s)) (throw-far-error
                         "Illegal list encoding"
                         :illegal-argument :illegal-list-encoding
                         (sym-map bencoded-str))
      :else (let [[obj rest-str] (parse-obj s)]
              (recur rest-str
                     (conj output obj))))))

(defn parse-map [bencoded-str]
  (loop [s (subs bencoded-str 1)
         output {}]
    (cond
      (= "e" (str (first s))) [output (subs s 1)]
      (zero? (count s)) (throw-far-error
                         "Illegal map encoding"
                         :illegal-argument :illegal-map-encoding
                         (sym-map bencoded-str))
      :else (let [[k rest-str] (parse-str s)
                  [v rest-str] (parse-obj rest-str)]
              (recur rest-str
                     (assoc output k v))))))

(defn parse-obj [bencoded-str]
  (let [tag (str (first bencoded-str))
        parse-fn (cond
                   (= "i" tag) parse-int
                   (= "l" tag) parse-list
                   (= "d" tag) parse-map
                   (contains? digits tag) parse-str
                   :else #(throw-far-error
                           "Unsupported encoding type"
                           :illegal-argument :unsupported-encoding
                           {:bencoded-str %}))]
    (parse-fn bencoded-str)))

(defn bencode->obj* [bencoded-str]
  (let [[obj rest-str] (parse-obj bencoded-str)]
    (if (zero? (count rest-str))
      obj
      (throw-far-error "Illegal encoding. Multiple top-level items"
                       :illegal-argument :multiple-top-level-items
                       (sym-map bencoded-str)))))
