(ns farbetter.roe.io-streams
  (:require
   [cljsjs.bytebuffer]
   [farbetter.roe.mutable-streams :as ms]
   [farbetter.utils :as u :refer [throw-far-error]]
   [goog.object])
  (:require-macros
   [farbetter.utils :refer [inspect sym-map]]))

(def ByteBuffer js/ByteBuffer)

(defrecord MutableOutputStream [buf]
  ms/OutputStream
  (get-size [this]
    (.-offset buf))

  (write-byte [this b]
    (.writeInt8 buf b)
    nil)

  (write-bytes [this bs num-bytes]
    (dotimes [n num-bytes]
      (ms/write-byte this (aget bs n))))

  (write-bytes-w-len-prefix [this bs]
    (let [num-bytes (.-length bs)]
      (ms/write-long-varint-zz this num-bytes)
      (ms/write-bytes this bs num-bytes)))

  (write-utf8-string [this s]
    (let [num-bytes (.calculateUTF8Bytes ByteBuffer s)]
      (ms/write-long-varint-zz this num-bytes)
      (.writeUTF8String buf s)
      nil))

  (write-long-varint-zz [this l]
    (.writeVarint64ZigZag buf l)
    nil)

  (write-float [this f]
    (.writeFloat32 buf f)
    nil)

  (write-double [this d]
    (.writeFloat64 buf d)
    nil)

  (to-byte-array [this]
    (.flip buf)
    (-> (.toArrayBuffer buf)
        (js/Int8Array.)))

  (to-b64-string [this]
    (.flip buf)
    (.toBase64 buf))

  (to-utf8-string [this]
    (.flip buf)
    (.toUTF8 buf)))

(defrecord MutableInputStream [buf]
  ms/InputStream
  (get-available-count [this]
    (.remaining buf))

  (read-byte [this]
    (.readInt8 buf))

  (read-bytes [this num-bytes]
    (let [ab (js/ArrayBuffer. num-bytes)
          view (js/Int8Array. ab)]
      (dotimes [n num-bytes]
        (->> (ms/read-byte this)
             (aset view n)))
      view))

  (read-len-prefixed-bytes [this]
    (let [num-bytes (ms/read-long-varint-zz this)]
      (ms/read-bytes this num-bytes)))

  (read-utf8-string [this]
    (let [num-bytes (-> (ms/read-long-varint-zz this)
                        (.toInt))]
      (.readUTF8String buf num-bytes (.-METRICS_BYTES ByteBuffer))))

  (read-long-varint-zz [this]
    (.readVarint64ZigZag buf))

  (read-float [this]
    (.readFloat32 buf))

  (read-double [this]
    (.readFloat64 buf))

  (mark [this read-limit]
    (.mark buf))

  (reset [this]
    ;; ByteBuffer's reset discards mark, we need to keep it for future use
    (let [mark (.-markedOffset buf)]
      (.reset buf)
      (goog.object/set buf "markedOffset" mark))))

(defn byte-array->mutable-input-stream [bs]
  (->MutableInputStream (.wrap ByteBuffer (.-buffer bs) "utf8" true)))

(defn b64-string->mutable-input-stream [s]
  (->MutableInputStream (ByteBuffer.fromBase64 s true)))

(defn make-mutable-output-stream []
  (->MutableOutputStream (ByteBuffer. 16 true)))
