(ns farbetter.roe.bencode
  (:require
   [clojure.string :as string :refer [join]]
   [farbetter.roe.io-streams :as ios]
   [farbetter.roe.mutable-streams :as ms]
   [farbetter.roe.schemas :refer [AvroSchema]]
   [farbetter.roe.transform :refer [transform-schema]]
   [farbetter.roe.utils :refer [#?@(:cljs [float?])]]
   [farbetter.utils :refer
    [byte-array? throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Special note:
;;
;; This implementation of bencode adds support for additional types:
;; - nil:   Represented as a single n "n" character.
;; - bool:  Represented as a "b" followed by 0 for false or 1 for true
;; - bytes: Reprepsented as an "x" followed by a bencoded base-64
;;          string representing the bytes.
;; - float: Represented as an "f" followed by the characters of the
;;          floating-point number, followed by an "x" character. The
;;          number may contain the letter "e" as is common in
;;          string representations of floating-point numbers. Minus "-"
;;          signs are also allowed.
;;          E.g.: "f3.14x", "f1.234e8x" "f1e-4x
;;          Due to inherent floating point representation
;;          inaccuracies, the decoded value may not exactly match
;;          the encoded value.
;;
;; In all other respects, this implementation follows the rules at:
;; https://wiki.theory.org/BitTorrentSpecification#Bencoding
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declare bencode->obj* obj->bencode* parse-obj)

(def digits #{"9" "3" "4" "8" "7" "5" "6" "1" "0" "2"})

;;;;;;;;; API ;;;;;;;;;

(s/defn obj->bencode :- s/Str
  [obj :- s/Any]
  (obj->bencode* obj))

(s/defn bencode->obj :- s/Any
  [s :- s/Str]
  (bencode->obj* s))

(s/defn ^:always-validate edn-schema->bencode :- s/Str
  ([schema :- AvroSchema]
   (edn-schema->bencode schema false))
  ([schema :- AvroSchema
    canonicalize? :- s/Bool]
   (let [transformed-schema (transform-schema schema :encode canonicalize?)]
     (obj->bencode transformed-schema))))

(s/defn bencode->edn-schema :- AvroSchema
  [s :- s/Str]
  (-> (bencode->obj s)
      (transform-schema :decode false)))

;;;;;;;;; Helper fns ;;;;;;;;;

(defn boolean? [obj]
  (or (true? obj)
      (false? obj)))

(defn byte-array->b64 [bytes]
  (let [mos (ios/make-mutable-output-stream)
        num-bytes (#?(:clj count :cljs .-length) bytes)]
    (ms/write-bytes mos bytes num-bytes)
    (ms/to-b64-string mos)))

(defn b64->byte-array [b64]
  (let [mis (ios/b64-string->mutable-input-stream b64)
        len (ms/get-available-count mis)]
    (ms/read-bytes mis len)))

(defn obj->bencode* [obj]
  (let [enc-int (fn [i]
                  (if (and (<= i 2147483647)
                           (>= i -2147483648))
                    (str "i" i "e")
                    (throw-far-error
                     (str "Absolute value of`" i
                          "` is too large for integer")
                     :illegal-argument :abs-value-too-large-for-integer
                     (sym-map i))))
        enc-float (fn [f]
                    (-> (str "f" f "x")
                        (string/replace #"\+" "")
                        (string/replace #"E" "e")))
        enc-str (fn [s]
                  (str (count s) ":" s))
        enc-list (fn [l]
                   (let [items (join (map obj->bencode l))]
                     (str "l" items "e")))
        enc-kv (fn [[k v]]
                 (if (string? k)
                   (-> (enc-str k)
                       (str (obj->bencode v)))
                   (throw-far-error
                    "Illegal non-string key in map type"
                    :illegal-argument :non-string-key
                    (sym-map k v))))
        enc-map (fn [m]
                  (let [kvs (seq (into (sorted-map) m))
                        enc-kvs (join (map enc-kv kvs))]
                    (str "d" enc-kvs "e")))
        enc-nil (fn [n]
                  "n")
        enc-bool (fn [b]
                   (let [v (case b
                             false 0
                             true 1)]
                     (str "b" v)))
        enc-bytes (fn [bytes]
                    (str "x" (enc-str (byte-array->b64 bytes))))
        enc-fn (cond
                 (integer? obj) enc-int
                 (float? obj) enc-float
                 (string? obj) enc-str
                 (sequential? obj) enc-list
                 (map? obj) enc-map
                 (nil? obj) enc-nil
                 (boolean? obj) enc-bool
                 (byte-array? obj) enc-bytes
                 :else #(throw-far-error "Unsupported type."
                                         :illegal-argument :unsupported-type
                                         {:value %}))]
    (enc-fn obj)))

(defn- str->int [s]
  (let [i #?(:clj
             (try
               (Integer/parseInt s)
               (catch Exception e
                 :not-an-integer))
             :cljs
             (let [i (js/parseInt s)]
               (if (js/isNaN i)
                 :not-an-integer
                 i)))]
    (when (= :not-an-integer i)
      (throw-far-error "Not an integer"
                       :illegal-argument :not-an-integer
                       (sym-map s)))
    i))

(defn- str->float [s]
  (let [f #?(:clj
             (try
               (Double/parseDouble s)
               (catch Exception e
                 :not-a-float))
             :cljs
             (let [f (js/parseFloat s)]
               (if (js/isNaN f)
                 :not-a-float
                 f)))]
    (when (= :not-a-float f)
      (throw-far-error "Not a floating-point number"
                       :illegal-argument :not-a-float
                       (sym-map s)))
    f))

(defn find-pos [s ch]
  (reduce (fn [acc c]
            (if (= ch (str c))
              (reduced acc)
              (inc acc)))
          0 s))

(defn parse-int [s]
  (let [end-pos (find-pos s "e")
        val (str->int (subs s 1 end-pos))]
    [val (subs s (inc end-pos))]))

(defn parse-float [s]
  (let [end-pos (find-pos s "x")
        val (str->float (subs s 1 end-pos))]
    [val (subs s (inc end-pos))]))

(defn parse-str [s]
  (let [colon-pos (find-pos s ":")
        len (str->int (subs s 0 colon-pos))
        start-pos (inc colon-pos)
        end-pos (+ start-pos len)
        val (subs s start-pos end-pos)]
    [val (subs s end-pos)]))

(defn parse-list [bencoded-str]
  (loop [s (subs bencoded-str 1)
         output []]
    (cond
      (= "e" (str (first s))) [output (subs s 1)]
      (zero? (count s)) (throw-far-error
                         "Illegal list encoding"
                         :illegal-argument :illegal-list-encoding
                         (sym-map bencoded-str))
      :else (let [[obj rest-str] (parse-obj s)]
              (recur rest-str
                     (conj output obj))))))

(defn parse-map [bencoded-str]
  (loop [s (subs bencoded-str 1)
         output {}]
    (cond
      (= "e" (str (first s))) [output (subs s 1)]
      (zero? (count s)) (throw-far-error
                         "Illegal map encoding"
                         :illegal-argument :illegal-map-encoding
                         (sym-map bencoded-str))
      :else (let [[k rest-str] (parse-str s)
                  [v rest-str] (parse-obj rest-str)]
              (recur rest-str
                     (assoc output k v))))))

(defn parse-nil [bencoded-str]
  [nil (subs bencoded-str 1)])

(defn parse-bool [bencoded-str]
  (let [ch (str (second bencoded-str))
        val (case ch
              "0" false
              "1" true
              (throw-far-error
               "Illegal bool encoding"
               :illegal-argument :illegal-bool-encoding
               (sym-map bencoded-str)))]
    [val (subs bencoded-str 2)]))

(defn parse-bytes [bencoded-str]
  (let [[b64-str rest-str] (parse-str (subs bencoded-str 1))
        bytes (b64->byte-array b64-str)]
    [bytes rest-str]))

(defn parse-obj [bencoded-str]
  (let [tag (str (first bencoded-str))
        parse-fn (cond
                   (= "i" tag) parse-int
                   (= "f" tag) parse-float
                   (= "l" tag) parse-list
                   (= "d" tag) parse-map
                   (= "n" tag) parse-nil
                   (= "b" tag) parse-bool
                   (= "x" tag) parse-bytes
                   (contains? digits tag) parse-str
                   :else #(throw-far-error
                           "Unsupported encoding type"
                           :illegal-argument :unsupported-encoding
                           {:bencoded-str %}))]
    (parse-fn bencoded-str)))

(defn bencode->obj* [bencoded-str]
  (let [[obj rest-str] (parse-obj bencoded-str)]
    (if (zero? (count rest-str))
      obj
      (throw-far-error "Illegal encoding. Multiple top-level items"
                       :illegal-argument :multiple-top-level-items
                       (sym-map bencoded-str)))))
