(ns farbetter.roe.transform
  (:require
   [clojure.string :as str]
   [farbetter.roe.schemas :refer [avro-primitive-type-names]]
   [farbetter.utils :refer [throw-far-error #?@(:clj [inspect sym-map])]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(defn transform-schema
  "Transform a clojure/script representation of the schema.
  If encode-or-decode is set to :encode, the input schema is assumed to be
  a clj-format schema, with keyword names which use  - instead of _.
  If encode-or-decode is set to :decode, the input schema is assumed to be
  an avro-format schema, with string names which use _ instead of -.
  The output schema will be the opposite type of the input schema.
  If canonicalize? is true, extra fields (doc and aliases) will be stripped."
  [s encode-or-decode canonicalize?]
  (when-not (contains? #{:encode :decode} encode-or-decode)
    (throw-far-error "Bad encode-or-decode value"
                     :illegal-argument :bad-encode-or-decode-value
                     (sym-map encode-or-decode)))
  (letfn [(get [m k]
            (case encode-or-decode
              :encode (m k)
              :decode (m (name k))))
          (update-key [m k f]
            (let [[source-key dest-key] (case encode-or-decode
                                          :encode [k (name k)]
                                          :decode [(name k) k])
                  source-val (m source-key)
                  dest-val (f source-val)]
              (-> (assoc m dest-key dest-val)
                  (dissoc m source-key))))
          (remove-key [m k]
            (let [source-key (case encode-or-decode
                               :encode k
                               :decode (name k))]
              (dissoc m source-key)))
          (name-fn [n]
            (case encode-or-decode
              :encode (-> (name n)
                          (str/replace \- \_))
              :decode (-> (str/replace n \_ \-)
                          (keyword))))
          (xf-schema [s]
            (cond
              (sequential? s) (xf-vec s)
              (map? s) (xf-typed s)
              :else (name-fn s)))
          (xf-vec [v]
            (mapv xf-schema v))
          (xf-prim-map [m]
            (m "type"))
          (xf-typed [m]
            (let [type (name (get m :type))
                  xf-fn (if (contains? avro-primitive-type-names type)
                          xf-prim-map
                          (case type
                            "record" xf-record
                            "enum" xf-enum
                            "fixed" xf-fixed
                            "array" xf-array
                            "map" xf-map))]
              (-> (update-key m :type name-fn)
                  (xf-fn))))
          (xf-name-doc-aliases [m]
            (let [m (update-key m :name xf-name)]
              (if canonicalize?
                (-> m
                    (remove-key :aliases)
                    (remove-key :doc))
                (cond-> m
                  (get m :aliases) (update-key :aliases #(mapv xf-name %))
                  (get m :doc) (update-key :doc identity)))))
          (xf-name [n]
            (name-fn n))
          (xf-field [f]
            (-> (xf-name-doc-aliases f)
                (update-key :type xf-schema)
                (update-key :default identity)
                (cond->
                    (get f :order) (update-key :order identity))))
          (xf-fields [fs]
            (mapv xf-field fs))
          (xf-record [r]
            (-> (xf-name-doc-aliases r)
                (update-key :fields xf-fields)))
          (xf-symbols [syms]
            (case encode-or-decode
              :encode (sort (map xf-name syms))
              :decode (into #{} (map xf-name syms))))
          (xf-enum [enum]
            (-> (xf-name-doc-aliases enum)
                (update-key :symbols xf-symbols)))
          (xf-array [a]
            (update-key a :items xf-schema))
          (xf-map [m]
            (update-key m :values xf-schema))
          (xf-fixed [f]
            (-> (xf-name-doc-aliases f)
                (update-key :size identity)))]
    (xf-schema s)))
