(ns farbetter.pete
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])))

(def TimeMs s/Num)
(def FnInfo {:f (s/=> s/Any)
             :interval-ms TimeMs})
(def RunTimeMap {TimeMs [FnInfo]})
(def RunTimeItem [(s/one TimeMs "time to run")
                   (s/one [FnInfo] "fn-infos")])

;;;;;;;;;;;;;;;;;;;; Protocol and Record Defs ;;;;;;;;;;;;;;;

(defprotocol IRepeater
  (start [this] "Restarts the calling of fns after stop has been called.")
  (stop [this] "Stops the calling of fns."))

(defrecord Repeater  [active?-atom]
  IRepeater

  (start [this]
    (reset! active?-atom true)
    nil)

  (stop [this]
    (reset! active?-atom false)
    nil))

;;;;;;;;;;;;;;;;;;;; Helper Fns ;;;;;;;;;;;;;;;;;;;;

(s/defn add-fn-info :- [FnInfo]
  [fn-infos :- (s/maybe [FnInfo])
   fn-info :- FnInfo]
  (-> (or fn-infos [])
      (conj fn-info)))

(s/defn handle-fn-info :- RunTimeMap
  [run-time :- TimeMs
   run? :- s/Bool
   rtm :- RunTimeMap
   fn-info :- FnInfo]
  (let [{:keys [f interval-ms]} fn-info
        next-run-time (+ run-time interval-ms)]
    (when run?
      (go-safe (f)))
    (update rtm next-run-time add-fn-info fn-info)))

(s/defn handle-run-time-item :- RunTimeMap
  [rtm :- RunTimeMap
   to-run :- RunTimeItem]
  (let [[run-time fn-infos] to-run
        rtm (dissoc rtm run-time)]
    (reduce (partial handle-fn-info run-time true)
            rtm fn-infos)))

(s/defn update-map-and-run-fns :- RunTimeMap
  [run-time-map :- RunTimeMap
   to-run-seq :- [RunTimeItem]]
  (reduce handle-run-time-item run-time-map to-run-seq))

(s/defn start-main-loop :- nil
  [active?-atom :- (s/atom s/Bool)
   run-time-map-atom :- (s/atom RunTimeMap)]
  (go-safe
   (while @active?-atom
     (when-let [to-run-seq (subseq @run-time-map-atom <=
                                   (u/get-current-time-ms))]
       (swap! run-time-map-atom update-map-and-run-fns to-run-seq))
     (ca/<! (ca/timeout 1))))
  nil)

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-repeater :- (s/protocol IRepeater)
  [fn-infos :- [FnInfo]]
  (let [active?-atom (atom true)
        f (partial handle-fn-info (u/get-current-time-ms) false)
        run-time-map (reduce f (sorted-map) fn-infos)
        run-time-map-atom (atom run-time-map)
        repeater (map->Repeater (sym-map active?-atom))]
    (start-main-loop active?-atom run-time-map-atom)
    repeater))
